use std::error::Error;
use std::fmt;

#[derive(Debug, PartialEq)]
pub enum GraphError {
    Regular(ErrorKind),
    Custom(String)
}

#[derive(Debug, PartialEq)]
pub enum ErrorKind {
    UnableCreateEdge(usize, usize),
    VertexNotFound(usize),
    TreeContainsCycle,
    ExistsCycle(usize)
}

impl fmt::Display for ErrorKind {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match *self {
            ErrorKind::UnableCreateEdge(from, to) => write!(f, "It is not possible to create an edge between these vertices ({}, {})", from, to),
            ErrorKind::VertexNotFound(id) => write!(f, "Vertex ({}) not found", id),
            ErrorKind::ExistsCycle(id) => write!(f, "Found cycle in the vertex ({})", id),
            ErrorKind::TreeContainsCycle => write!(f, "Found cycle. Tree must not contain cycle"),
        }
    }
}

impl fmt::Display for GraphError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match *self {
            GraphError::Regular(ref err) => write!(f, "A regular error occurred. {}", err),
            GraphError::Custom(ref err) => write!(f, "A custom error occurred. {}", err),
        }
    }
}

impl Error for GraphError {

}
