use std::fmt::Write;
use std::rc::Rc;

use regex::Regex;

use crate::error::Result;
use crate::vm::{Numeric, Value, VM};

// TODO: Refactor this into full regex parser,
// regex crate does not support recursion needed for '%bxy' patterns
fn build_pattern(pat: &str) -> Regex {
    let mut pat = pat.chars().peekable();
    let mut re = String::new();

    let escaped = |re: &mut String, c| match c {
        'a' => re.push_str(r"[\p{L}\p{Nl}]"),
        'c' => re.push_str(r"\p{Cc}"),
        'd' => re.push_str(r"\p{Nd}"),
        'g' => re.push_str(r"[\p{Z}\p{C}]"),
        'l' => re.push_str(r"\p{Ll}"),
        'p' => re.push_str(r"\p{P}"),
        's' => re.push_str(r"[\p{Z}\t\r\n\v\f]"),
        'u' => re.push_str(r"\p{Lu}"),
        'w' => re.push_str(r"[\p{L}\p{Nl}\p{Nd}]"),
        'x' => re.push_str(r"[0-9A-Fa-f]"),
        c => {
            if matches!(
                c,
                '\\' | '.' | '+' | '*' | '?' | '(' | ')' | '|' | '[' | ']' | '{' | '}' | '^' | '$'
            ) {
                re.push('\\');
            }
            re.push(c);
        }
    };

    while let Some(char) = pat.next() {
        match char {
            '%' => escaped(&mut re, pat.next().unwrap()),
            '[' => {
                re.push('[');
                loop {
                    match pat.next().unwrap() {
                        ']' => {
                            re.push(']');
                            break;
                        }
                        '%' => escaped(&mut re, pat.next().unwrap()),
                        c => re.push(c),
                    }
                }
            }
            '-' => re.push_str("*?"),
            c => re.push(c),
        }
    }

    Regex::new(&re).unwrap()
}

pub(super) fn byte(s: Value, i: Value, j: Value) -> Result<Value> {
    let mut out = Vec::new();

    let s = s.to_string_coerce()?;
    let s = s.as_bytes();
    let i = match i {
        Value::Nil => 1,
        v => v.to_number_coerce()?.coerce_int()?,
    };
    let j = match j {
        Value::Nil => i,
        v => v.to_number_coerce()?.coerce_int()?,
    };
    for i in (i - 1)..j {
        if i >= 0 {
            if let Some(b) = s.get(i as usize) {
                out.push(Value::int(*b as i64));
            }
        }
    }

    Ok(Value::Mult(out))
}

pub(super) fn char(args: Vec<Value>) -> Result<Value> {
    let mut out = String::new();

    for arg in args {
        let num = arg.to_number_coerce()?.coerce_int()?;
        if num < u8::MIN as i64 || (u8::MAX as i64) < num {
            panic!("char out of range")
        }
        out.push(num as u8 as char);
    }

    Ok(Value::String(out))
}

pub(super) fn find(s: Value, pattern: Value) -> Result<Value> {
    let s = s.to_string()?;
    let pattern = pattern.to_string()?;

    let re = build_pattern(pattern);

    Ok(match re.find(s) {
        Some(m) => Value::Mult(vec![
            Value::Number(Numeric::from_usize_try_int(m.start() + 1)),
            Value::Number(Numeric::from_usize_try_int(m.end())),
        ]),
        None => Value::Nil,
    })
}

#[derive(Default)]
struct FormatOpt {
    conv: char,               // %{specifier}
    just_left: bool,          // %-s
    sign_force: bool,         // %+s
    sign_space: bool,         // % s
    format_alt: bool,         // %#s
    pad_zeroes: bool,         // %02s
    min_width: Option<usize>, // %2s
    precision: Option<usize>, // %.2s
}

pub(super) fn format(formatstring: Value, mut args: Vec<Value>) -> Result<Value> {
    let format = formatstring.to_string()?;
    let mut format = format.chars().peekable();
    let mut out = String::new();

    while let Some(char) = format.next() {
        if !matches!(char, '%') {
            out.push(char);
            continue;
        }

        // Read modifiers
        let mut opt = FormatOpt::default();
        loop {
            match format.peek().copied() {
                Some(c) if matches!(c, '-' | '+' | ' ' | '#' | '0') => {
                    format.next();
                    match c {
                        '-' => opt.just_left = true,
                        '+' => opt.sign_force = true,
                        ' ' => opt.sign_space = true,
                        '#' => opt.format_alt = true,
                        '0' => opt.pad_zeroes = true,
                        _ => unreachable!(),
                    }
                }
                _ => break,
            }
        }
        opt.min_width = {
            let mut digits = vec![];
            loop {
                match format.peek().copied() {
                    Some(c) if matches!(c, '0'..='9') => {
                        format.next();
                        digits.push(c);
                    }
                    _ => break,
                }
            }
            if digits.is_empty() {
                None
            } else {
                Some(digits.iter().collect::<String>().parse().unwrap())
            }
        };
        opt.precision = match format.peek() {
            Some('.') => {
                format.next();
                let mut digits = vec![];
                loop {
                    match format.peek().copied() {
                        Some(c) if matches!(c, '0'..='9') => {
                            format.next();
                            digits.push(c);
                        }
                        _ => break,
                    }
                }
                if digits.is_empty() {
                    panic!()
                } else {
                    Some(digits.iter().collect::<String>().parse().unwrap())
                }
            }
            _ => None,
        };

        // Read conversion specifier
        opt.conv = format.next().unwrap();
        // Shortcut, because no need to take an argument
        if opt.conv == '%' {
            write!(out, "%")?;
            continue;
        }

        if args.is_empty() {
            panic!("not enough arguments");
        }
        let arg = args.remove(0);

        if matches!(opt.conv, 'q')
            && (opt.just_left
                || opt.sign_force
                || opt.sign_space
                || opt.format_alt
                || opt.pad_zeroes
                || opt.min_width.is_some()
                || opt.precision.is_some())
        {
            panic!("format specifier q supports no modifiers")
        }

        let formatted = format_arg(&opt, arg)?;

        let conv_hex = matches!(opt.conv, 'A' | 'a' | 'p');
        let conv_number = matches!(
            opt.conv,
            'd' | 'i' | 'o' | 'u' | 'X' | 'x' | 'E' | 'e' | 'f' | 'G' | 'g'
        );
        let min_width = opt.min_width.unwrap_or(0);
        if opt.just_left {
            write!(out, "{:<1$}", formatted, min_width)?;
        } else if opt.pad_zeroes && conv_hex {
            let ind = formatted.find(|c| c == 'x' || c == 'X').unwrap();
            write!(
                out,
                "{}{:0>2$}",
                &formatted[..ind + 1],
                &formatted[ind + 1..],
                min_width.saturating_sub(ind + 1)
            )?;
        } else if opt.pad_zeroes && conv_number {
            write!(out, "{:0>1$}", formatted, min_width)?;
        } else {
            write!(out, "{:>1$}", formatted, min_width)?;
        }
    }

    Ok(Value::String(out))
}

fn format_arg(opt: &FormatOpt, arg: Value) -> Result<String> {
    match opt.conv {
        's' => {
            let mut str = format!("{}", arg);
            if let Some(precision) = opt.precision {
                str.truncate(precision);
            }
            Ok(str)
        }
        'c' => Ok(format!(
            "{}",
            arg.to_number_coerce()?.to_int()? as u8 as char
        )),
        'd' | 'i' => {
            let n = arg.to_number_coerce()?.to_int()?;
            let mut prec = opt.precision.unwrap_or(1);
            let prefix = if n < 0 {
                prec += 1;
                ""
            } else if opt.sign_force {
                "+"
            } else if opt.sign_space {
                " "
            } else {
                ""
            };
            let digits = format!("{:01$}", n, prec);
            Ok(if prec == 0 && n == 0 {
                "".to_string() // Don't output anything
            } else {
                format!("{}{}", prefix, digits)
            })
        }
        'o' => {
            let n = arg.to_number_coerce()?.to_int()? as u64;
            let prec = opt.precision.unwrap_or(1);
            Ok(if prec == 0 && n == 0 {
                (if opt.format_alt { "0" } else { "" }).to_string()
            } else if opt.format_alt && n > 0 {
                format!("0{:01$o}", n, prec)
            } else {
                format!("{:01$o}", n, prec)
            })
        }
        'u' | 'X' | 'x' => {
            let n = arg.to_number_coerce()?.to_int()? as u64;
            let prec = opt.precision.unwrap_or(1);
            Ok(if prec == 0 && n == 0 {
                "".to_string()
            } else {
                match opt.conv {
                    'u' => format!("{:01$}", n, prec),
                    'X' => format!("{:01$X}", n, prec),
                    'x' => format!("{:01$x}", n, prec),
                    _ => unreachable!(),
                }
            })
        }
        'A' | 'a' => {
            let n = arg.to_number_coerce()?.to_float();
            let cap = matches!(opt.conv, 'A');
            let mut out = String::new();
            // Masks
            const F64_MASK_SIGN: u64 = 0x8000_0000_0000_0000;
            const F64_MASK_EXP: u64 = 0x7ff0_0000_0000_0000;
            const F64_MASK_FRACT: u64 = 0x000f_ffff_ffff_ffff;
            // Parts
            let bits = n.to_bits();
            let neg = bits & F64_MASK_SIGN != 0;
            let exp = (bits & F64_MASK_EXP) >> 52;
            let fract = bits & F64_MASK_FRACT;
            // Write sign
            if neg {
                write!(out, "-")?;
            } else if opt.sign_force {
                write!(out, "+")?;
            } else if opt.sign_space {
                write!(out, " ")?;
            }
            // Write digits and exponent
            if exp == 0x7ff {
                if fract == 0 {
                    write!(out, "{}", if cap { "INF" } else { "inf" })?;
                } else {
                    write!(out, "{}", if cap { "NAN" } else { "nan" })?;
                }
            } else {
                write!(out, "0{}", if cap { 'X' } else { 'x' })?;
                let mut fract_str = if cap {
                    format!("{:X}", fract)
                } else {
                    format!("{:x}", fract)
                };
                let fract_str = if let Some(prec) = opt.precision {
                    if fract_str.len() >= prec {
                        &fract_str[..prec]
                    } else {
                        fract_str.push_str(&"0".repeat(prec - fract_str.len()));
                        fract_str.as_str()
                    }
                } else {
                    fract_str.trim_end_matches('0')
                };
                let exp = if exp == 0x0 {
                    if fract == 0 {
                        write!(out, "0")?;
                        0i64
                    } else {
                        write!(out, "0.{}", fract_str)?;
                        -1022
                    }
                } else {
                    if fract == 0 && !opt.format_alt {
                        write!(out, "1")?;
                    } else {
                        write!(out, "1.{}", fract_str)?;
                    }
                    exp as i64 - 1023
                };
                write!(out, "{}", if cap { 'P' } else { 'p' })?;
                write!(out, "{}", exp)?;
            }
            Ok(out)
        }
        'E' | 'e' | 'f' => {
            let n = arg.to_number_coerce()?.to_float();
            let prec = opt.precision.unwrap_or(6);
            let prefix = if n.is_sign_negative() {
                "-"
            } else if opt.sign_force {
                "+"
            } else if opt.sign_space {
                " "
            } else {
                ""
            };
            let digits = match opt.conv {
                'E' => format!("{:.1$E}", n.abs(), prec),
                'e' => format!("{:.1$e}", n.abs(), prec),
                'f' => format!("{:.1$}", n.abs(), prec),
                _ => unreachable!(),
            };
            let digits = if n.abs().fract() == 0.0 && opt.format_alt {
                match opt.conv {
                    'E' | 'e' => {
                        let pos = digits.find(opt.conv).unwrap();
                        format!("{}.{}", &digits[..pos], &digits[pos..])
                    }
                    'f' => {
                        format!("{}.", digits)
                    }
                    _ => unreachable!(),
                }
            } else {
                digits
            };
            Ok(format!("{}{}", prefix, digits))
        }
        'G' | 'g' => {
            let n = arg.to_number_coerce()?.to_float();
            let prec = opt.precision.unwrap_or(6).max(1);
            let e = n.abs().log10().floor();
            Ok(if prec as f64 > e && e >= -4. {
                let prec = prec - 1 - e as usize;
                let digits = format_arg(
                    &FormatOpt {
                        conv: 'f',
                        sign_force: opt.sign_force,
                        sign_space: opt.sign_space,
                        precision: Some(prec),
                        ..Default::default()
                    },
                    Value::float(n),
                )?;
                if opt.format_alt {
                    digits
                } else {
                    digits
                        .trim_end_matches('0')
                        .trim_end_matches('.')
                        .to_string()
                }
            } else {
                let prec = prec - 1;
                let format = if matches!(opt.conv, 'G') { 'E' } else { 'e' };
                let digits = format_arg(
                    &FormatOpt {
                        conv: format,
                        sign_force: opt.sign_force,
                        sign_space: opt.sign_space,
                        precision: Some(prec),
                        ..Default::default()
                    },
                    Value::float(n),
                )?;
                if opt.format_alt {
                    digits
                } else {
                    let mut digits = digits;
                    let exp = digits.split_off(digits.find(format).unwrap());
                    format!(
                        "{}{}",
                        digits.trim_end_matches('0').trim_end_matches('.'),
                        exp
                    )
                }
            })
        }
        'p' => {
            let digits = match arg {
                Value::String(s) => format!("{:?}", &s as *const _),
                Value::Func(f) => format!("{:?}", Rc::as_ptr(&f)),
                Value::Table(r) => format!("{:?}", Rc::as_ptr(&r)),
                _ => "(null)".to_string(),
            };
            let prec = opt.precision.unwrap_or(0);
            let digits = if digits.len() < prec {
                format!(
                    "{}{}{}",
                    &digits[..2],
                    "0".repeat(prec - digits.len()),
                    &digits[2..]
                )
            } else {
                digits
            };
            let prefix = if digits.starts_with('(') {
                ""
            } else if opt.sign_force {
                "+"
            } else if opt.sign_space {
                " "
            } else {
                ""
            };
            Ok(format!("{}{}", prefix, digits))
        }
        'q' => match arg.into_single() {
            Value::Nil => Ok("nil".to_string()),
            Value::Bool(b) => Ok(format!("{}", b)),
            Value::Number(n) => match n {
                Numeric::Integer(i) => Ok(format!("{}", i)),
                Numeric::Float(f) => format_arg(
                    &FormatOpt {
                        conv: 'a',
                        ..Default::default()
                    },
                    Value::float(f),
                ),
            },
            Value::String(s) => Ok(format!(
                "\"{}\"",
                s.replace("\"", "\\\"").replace("\n", "\\\n")
            )),
            _ => panic!("arg not a literal"),
        },
        _ => panic!("invalid format string"),
    }
}

pub(super) fn gsub(vm: &mut VM, s: Value, pattern: Value, repl: Value, n: Value) -> Result<Value> {
    let s = s.to_string()?;
    let pattern = build_pattern(pattern.to_string()?);
    let n = match n {
        Value::Nil => 0,
        v => v.to_number_coerce()?.to_int()? as usize,
    };
    let len = pattern.find_iter(s).count();
    let replaced = match repl {
        Value::String(repl) => {
            let repl_regex = Regex::new(r"%(\d+)").unwrap();
            let res = pattern.replacen(s, n, |cap_outer: &regex::Captures| {
                repl_regex.replace_all(&repl, |cap_inner: &regex::Captures| {
                    &cap_outer[cap_inner[1].parse::<usize>().unwrap()]
                })
            });
            res.to_string()
        }
        Value::Func(func) => {
            let res = pattern.replacen(s, n, |cap: &regex::Captures| {
                let caps = if cap.len() == 1 {
                    vec![Value::String(cap[0].to_string())]
                } else {
                    (1..cap.len())
                        .map(|n| Value::String(cap[n].to_string()))
                        .collect()
                };

                // (Ab)use protected calls for this
                match vm.call_protected(func.clone(), caps) {
                    Ok(ret) => {
                        format!("{}", ret.into_single())
                    }
                    Err(e) => panic!("{}", e),
                }
            });
            res.to_string()
        }
        Value::Table(tbl) => {
            let tbl = tbl.borrow();
            let res = pattern.replacen(s, n, |cap: &regex::Captures| {
                format!("{}", tbl.get(&Value::String(cap[1].to_string())))
            });
            res.to_string()
        }
        _ => panic!("invalid value"),
    };
    Ok(Value::Mult(vec![
        Value::String(replaced),
        Value::int(len as i64),
    ]))
}

pub(super) fn rep(s: Value, n: Value, sep: Value) -> Result<Value> {
    let mut out = String::new();

    let s = s.to_string_coerce()?;
    let n = n.to_number_coerce()?.coerce_int()?;
    let sep = match sep {
        Value::Nil => "".to_string(),
        v => v.to_string_coerce()?,
    };

    for i in 0..n {
        if i > 0 {
            write!(out, "{}", sep)?;
        }
        write!(out, "{}", s)?;
    }

    Ok(Value::String(out))
}
