use std::cell::RefCell;
use std::rc::Rc;

mod basic;
mod math;
mod string;

use crate::vm::{FuncBuiltin, FuncDef, Table, Value};

macro_rules! def {
    ( $env:expr, $name:ident ( $($args:tt)* ) $func:expr ) => {{
        $env.set(
            Value::String(stringify!($name).to_string()),
            Value::Func(Rc::new(FuncDef::Builtin(FuncBuiltin{
                name: stringify!($name),
                func: _func!( $func, ( $($args)* ) ),
            }))),
        ).unwrap();
    }};
    ( $env:expr, mod $name:ident ) => {{
        $env.set(
            Value::String(stringify!($name).to_string()),
            Value::Table(Rc::new(RefCell::new(Table::default())))
        ).unwrap();
    }};
    ( $env:expr, const $module:ident.$name:ident, $val:expr ) => {{
        let module = match $env.get(&Value::String(stringify!($module).to_string())) {
            Value::Table(t) => t,
            _ => unreachable!(),
        };
        module.borrow_mut().set(
            Value::String(stringify!($name).to_string()),
            $val,
        ).unwrap();
    }};
    ( $env:expr, $module:ident.$name:ident ( $($args:tt)* ) $imp:expr ) => {{
        let module = match $env.get(&Value::String(stringify!($module).to_string())) {
            Value::Table(t) => t,
            _ => unreachable!(),
        };
        def!( module.borrow_mut(), $name ( $($args)* ) $imp );
    }};
}

macro_rules! _func {
    ( $func:expr, ( vm, $($params:tt)* ) ) => {
        |vm, args| {
            let mut args = args.into_iter();
            _func!( &mut args, $func, ( vm ) | ( $($params)* ) )
        }
    };
    ( $func:expr, ( $($params:tt)* ) ) => {
        |_, args| {
            let mut args = args.into_iter();
            _func!( &mut args, $func, () | ( $($params)* ) )
        }
    };
    ( $iter:expr, $func:expr, () | ( $param:ident, $($params:tt)* ) ) => {{
        let $param = $iter.next().unwrap_or(Value::Nil);
        _func!( $iter, $func, ( $param ) | ( $($params)* ) )
    }};
    ( $iter:expr, $func:expr, ( $($args:tt)* ) | ( $param:ident, $($params:tt)* ) ) => {{
        let $param = $iter.next().unwrap_or(Value::Nil);
        _func!( $iter, $func, ( $($args)*, $param ) | ( $($params)* ) )
    }};
    ( $iter:expr, $func:expr, () | ( $param:ident ) ) => {{
        let $param = $iter.next().unwrap_or(Value::Nil);
        _func!( $iter, $func, ( $param ) | () )
    }};
    ( $iter:expr, $func:expr, ( $($args:tt)* ) | ( $param:ident ) ) => {{
        let $param = $iter.next().unwrap_or(Value::Nil);
        _func!( $iter, $func, ( $($args)*, $param ) | () )
    }};
    ( $iter:expr, $func:expr, () | (...) ) => {
        _func!( $iter, $func, ( $iter.collect() ) | ())
    };
    ( $iter:expr, $func:expr, ( $($args:tt)* ) | (...) ) => {{
        _func!( $iter, $func, ( $($args)*, $iter.collect() ) | ())
    }};
    ( $iter:expr, $func:expr, ( $($args:tt)* ) | () ) => {
        $func( $($args)* )
    };
}

pub(super) fn load(env: &mut Table) {
    def!(env, assert(v, message) basic::assert);
    def!(env, error(message, level) basic::error);
    def!(env, ipairs(vm, t) basic::ipairs);
    def!(env, __ipairs_next(table, index) basic::__ipairs_next);
    def!(env, load(vm, chunk) basic::load);
    def!(env, next(table, index) basic::next);
    def!(env, pairs(vm, t) basic::pairs);
    def!(env, pcall(vm, f, ...) basic::pcall);
    def!(env, print(vm, ...) basic::print);
    def!(env, rawget(table, index) basic::rawget);
    def!(env, rawlen(v) basic::rawlen);
    def!(env, rawset(table, index, value) basic::rawset);
    def!(env, select(index, ...) basic::select);
    def!(env, tonumber(v, base) basic::tonumber);
    def!(env, tostring(v) basic::tostring);
    def!(env, type(v) basic::r#type);

    def!(env, mod math);
    def!(env, math.abs(x) math::abs);
    def!(env, math.acos(x) math::acos);
    def!(env, math.asin(x) math::asin);
    def!(env, math.atan(y, x) math::atan);
    def!(env, math.ceil(x) math::ceil);
    def!(env, math.cos(x) math::cos);
    def!(env, math.deg(x) math::deg);
    def!(env, math.exp(x) math::exp);
    def!(env, math.floor(x) math::floor);
    def!(env, math.fmod(x, y) math::fmod);
    def!(env, const math.huge, Value::float(math::HUGE));
    def!(env, math.log(x, base) math::log);
    def!(env, math.max(x, ...) math::max);
    def!(env, const math.maxinteger, Value::int(math::MAX_INTEGER));
    def!(env, math.min(x, ...) math::min);
    def!(env, const math.mininteger, Value::int(math::MIN_INTEGER));
    def!(env, math.modf(x) math::modf);
    def!(env, const math.pi, Value::float(math::PI));
    def!(env, math.rad(x) math::rad);
    def!(env, math.random(vm, m, n, ...) math::random);
    def!(env, math.randomseed(vm, x, y) math::randomseed);
    def!(env, math.sin(x) math::sin);
    def!(env, math.sqrt(x) math::sqrt);
    def!(env, math.tan(x) math::tan);
    def!(env, math.tointeger(x) math::tointeger);
    def!(env, math.type(x) math::r#type);
    def!(env, math.ult(m, n) math::ult);

    def!(env, mod string);
    def!(env, string.byte(s, i, j) string::byte);
    def!(env, string.char(...) string::char);
    def!(env, string.find(s, pattern) string::find);
    def!(env, string.format(formatstring, ...) string::format);
    def!(env, string.gsub(vm, s, pattern, repl, n) string::gsub);
    def!(env, string.rep(s, n, sep) string::rep);
}
