use std::cmp::Ordering;
use std::convert::TryInto;

use rand::{Rng, SeedableRng};
use rand_xoshiro::Xoshiro256StarStar;

use crate::error::{LuaError, Result};
use crate::vm::{BinOp, Numeric, Value, ValueType, VM};

pub(super) fn abs(x: Value) -> Result<Value> {
    Ok(Value::Number(match x.to_number_coerce()? {
        Numeric::Integer(i) => Numeric::Integer(i.wrapping_abs()),
        Numeric::Float(f) => Numeric::Float(f.abs()),
    }))
}

pub(super) fn acos(x: Value) -> Result<Value> {
    Ok(Value::float(x.to_number_coerce()?.to_float().acos()))
}

pub(super) fn asin(x: Value) -> Result<Value> {
    Ok(Value::float(x.to_number_coerce()?.to_float().asin()))
}

pub(super) fn atan(y: Value, x: Value) -> Result<Value> {
    let y = y.to_number_coerce()?.to_float();
    Ok(Value::float(match x {
        Value::Number(n) => y.atan2(n.to_float()),
        Value::Nil => y.atan(),
        v => return err!(LuaError::ExpectedType(ValueType::Number, v.value_type())),
    }))
}

pub(super) fn ceil(x: Value) -> Result<Value> {
    Ok(Value::Number(match x.to_number_coerce()? {
        Numeric::Integer(n) => Numeric::Integer(n),
        Numeric::Float(f) => Numeric::from_f64_try_int(f.ceil()),
    }))
}

pub(super) fn cos(x: Value) -> Result<Value> {
    Ok(Value::float(x.to_number_coerce()?.to_float().cos()))
}

pub(super) fn deg(x: Value) -> Result<Value> {
    Ok(Value::float(x.to_number_coerce()?.to_float().to_degrees()))
}

pub(super) fn exp(x: Value) -> Result<Value> {
    Ok(Value::float(x.to_number_coerce()?.to_float().exp()))
}

pub(super) fn floor(x: Value) -> Result<Value> {
    Ok(Value::Number(match x.to_number_coerce()? {
        Numeric::Integer(n) => Numeric::Integer(n),
        Numeric::Float(f) => Numeric::from_f64_try_int(f.floor()),
    }))
}

pub(super) fn fmod(x: Value, y: Value) -> Result<Value> {
    let x = x.to_number_coerce()?;
    let y = y.to_number_coerce()?;
    Ok(Value::Number(x.rem(&y)?))
}

pub(super) const HUGE: f64 = std::f64::INFINITY;

pub(super) fn log(x: Value, base: Value) -> Result<Value> {
    let x = x.to_number_coerce()?.to_float();
    Ok(Value::float(match base {
        Value::Number(n) => x.log(n.to_float()),
        Value::Nil => x.ln(),
        v => return err!(LuaError::ExpectedType(ValueType::Number, v.value_type())),
    }))
}

pub(super) fn max(x: Value, rest: Vec<Value>) -> Result<Value> {
    let mut max = match x {
        Value::Nil => return err!(LuaError::ExpectedValue),
        v => v,
    };
    for val in rest.into_iter() {
        if max.bin_op(BinOp::Lt, &val)?.is_truthy() {
            max = val;
        }
    }
    Ok(max)
}

pub(super) const MAX_INTEGER: i64 = std::i64::MAX;

pub(super) fn min(x: Value, rest: Vec<Value>) -> Result<Value> {
    let mut min = match x {
        Value::Nil => return err!(LuaError::ExpectedValue),
        v => v,
    };
    for val in rest.into_iter() {
        if val.bin_op(BinOp::Lt, &min)?.is_truthy() {
            min = val;
        }
    }
    Ok(min)
}

pub(super) const MIN_INTEGER: i64 = std::i64::MIN;

pub(super) fn modf(x: Value) -> Result<Value> {
    let x = x.to_number_coerce()?.to_float();
    Ok(Value::Mult(vec![
        Value::Number(Numeric::from_f64_try_int(x.trunc())),
        Value::float(if x.is_infinite() { 0.0 } else { x.fract() }),
    ]))
}

pub(super) const PI: f64 = std::f64::consts::PI;

pub(super) fn rad(x: Value) -> Result<Value> {
    Ok(Value::float(x.to_number_coerce()?.to_float().to_radians()))
}

pub(super) fn random(vm: &mut VM, m: Value, n: Value, args: Vec<Value>) -> Result<Value> {
    if !args.is_empty() {
        return err!(LuaError::ArgumentCount);
    }
    Ok(match m {
        Value::Number(lower) => {
            let lower = lower.to_int()?;
            match n {
                Value::Number(upper) => {
                    let range = lower..=upper.to_int()?;
                    if range.is_empty() {
                        return err!(LuaError::EmptyInterval);
                    }
                    Value::int(vm.rng.gen_range(range))
                }
                Value::Nil => match lower.cmp(&0) {
                    Ordering::Greater => Value::int(vm.rng.gen_range(1..=lower)),
                    Ordering::Equal => Value::int(vm.rng.gen()),
                    Ordering::Less => panic!(),
                },
                v => return err!(LuaError::ExpectedType(ValueType::Number, v.value_type())),
            }
        }
        Value::Nil => Value::float(vm.rng.gen()),
        v => return err!(LuaError::ExpectedType(ValueType::Number, v.value_type())),
    })
}

pub(super) fn randomseed(vm: &mut VM, x: Value, y: Value) -> Result<Value> {
    let x = match x {
        Value::Number(n) => n.to_int()?,
        Value::Nil => random(vm, Value::int(0), Value::Nil, vec![])
            .unwrap()
            .to_number()?
            .to_int()?,
        v => return err!(LuaError::ExpectedType(ValueType::Number, v.value_type())),
    };
    let y = match y {
        Value::Number(n) => n.to_int()?,
        Value::Nil => 0,
        v => return err!(LuaError::ExpectedType(ValueType::Number, v.value_type())),
    };
    let seed: Vec<u8> = [
        x.to_ne_bytes(),
        0xffu64.to_ne_bytes(),
        y.to_ne_bytes(),
        0u64.to_ne_bytes(),
    ]
    .concat();
    vm.rng = Xoshiro256StarStar::from_seed(seed.try_into().unwrap());
    // Discard some values (reference implementation does this)
    for _ in 0..16 {
        random(vm, Value::int(0), Value::Nil, vec![]).unwrap();
    }
    Ok(Value::Mult(vec![Value::int(x), Value::int(y)]))
}

pub(super) fn sin(x: Value) -> Result<Value> {
    Ok(Value::float(x.to_number_coerce()?.to_float().sin()))
}

pub(super) fn sqrt(x: Value) -> Result<Value> {
    Ok(Value::float(x.to_number_coerce()?.to_float().sqrt()))
}

pub(super) fn tan(x: Value) -> Result<Value> {
    Ok(Value::float(x.to_number_coerce()?.to_float().tan()))
}

pub(super) fn tointeger(x: Value) -> Result<Value> {
    Ok(match x.to_number_coerce().and_then(|n| n.coerce_int()) {
        Ok(n) => Value::int(n),
        Err(..) => Value::Nil,
    })
}

pub(super) fn r#type(x: Value) -> Result<Value> {
    Ok(match x {
        Value::Number(n) => Value::String(
            match n {
                Numeric::Integer(..) => "integer",
                Numeric::Float(..) => "float",
            }
            .to_string(),
        ),
        _ => Value::Nil,
    })
}

pub(super) fn ult(m: Value, n: Value) -> Result<Value> {
    Ok(Value::Bool(
        (m.to_number_coerce()?.to_int()? as u64) < n.to_number_coerce()?.to_int()? as u64,
    ))
}
