use std::cmp::Ordering;
use std::rc::Rc;

use crate::error::{Error, LuaError, Result};
use crate::vm::{Numeric, Value, ValueType, VM};

pub(super) fn assert(v: Value, error: Value) -> Result<Value> {
    if v.is_falsy() {
        return match error {
            Value::Nil => err!(LuaError::AssertFailed),
            v => err!(LuaError::Custom(format!("{}", v))),
        };
    }

    Ok(Value::Mult(vec![v, error]))
}

pub(super) fn error(message: Value, _level: Value) -> Result<Value> {
    err!(LuaError::Custom(format!("{}", message)))
}

pub(super) fn ipairs(vm: &mut VM, t: Value) -> Result<Value> {
    let iter = vm
        .env
        .borrow()
        .get(&Value::String("__ipairs_next".to_string()));
    Ok(Value::Mult(vec![iter, t, Value::int(0)]))
}

pub(super) fn __ipairs_next(table: Value, index: Value) -> Result<Value> {
    let table = table.to_table()?;
    let table = table.borrow();
    let next = match index {
        Value::Number(Numeric::Integer(n)) => Value::int(n + 1),
        _ => panic!("invalid state"),
    };
    Ok(Value::Mult(match table.get(&next) {
        Value::Nil => vec![Value::Nil],
        val => vec![next, val],
    }))
}

pub(super) fn load(vm: &mut VM, chunk: Value) -> Result<Value> {
    let string = chunk.to_string()?;

    let func = vm.load_str(string)?;

    Ok(Value::Func(Rc::new(func)))
}

pub(super) fn next(table: Value, index: Value) -> Result<Value> {
    let table = table.to_table()?;
    let mut table = table.borrow_mut();
    let next = match index.into_single() {
        Value::Nil => table.init_next_keys(),
        v => table.next_key(v),
    };
    let val = table.get(&next);
    Ok(Value::Mult(vec![next, val]))
}

pub(super) fn pairs(vm: &mut VM, t: Value) -> Result<Value> {
    let iter = vm.env.borrow().get(&Value::String("next".to_string()));
    Ok(Value::Mult(vec![iter, t, Value::Nil]))
}

pub(super) fn pcall(vm: &mut VM, f: Value, mut args: Vec<Value>) -> Result<Value> {
    let func = f.to_func()?;

    let last_arg = args.pop().map(Value::into_vec).unwrap_or_default();
    let args = args
        .into_iter()
        .map(Value::into_single)
        .chain(last_arg.into_iter())
        .collect();

    match vm.call_protected(func, args) {
        Ok(ret) => {
            let mut ret = ret.into_vec();
            ret.insert(0, Value::Bool(true));
            Ok(Value::Mult(ret))
        }
        Err(e) => match e {
            Error::Lua { typ, .. } => Ok(Value::Mult(vec![
                Value::Bool(false),
                Value::String(format!("{}", typ)),
            ])),
            _ => panic!(),
        },
    }
}

pub(super) fn print(vm: &mut VM, args: Vec<Value>) -> Result<Value> {
    let mut iter = args.iter();
    if let Some(first) = iter.next() {
        write!(vm.out, "{}", first)?;
    }
    for val in iter {
        write!(vm.out, " {}", val)?;
    }

    writeln!(vm.out)?;
    vm.out.flush().unwrap();
    Ok(Value::empty())
}

pub(super) fn rawget(table: Value, index: Value) -> Result<Value> {
    let table = table.to_table()?;
    let table = table.borrow();
    Ok(table.get(&index))
}

pub(super) fn rawlen(v: Value) -> Result<Value> {
    let len = match v {
        Value::String(s) => s.len(),
        Value::Table(tbl) => tbl.borrow().len(),
        _ => panic!("expected table or string"),
    };
    Ok(Value::int(len as i64))
}

pub(super) fn rawset(table: Value, index: Value, value: Value) -> Result<Value> {
    let table = table.to_table()?;
    table.borrow_mut().set(index, value)?;
    Ok(Value::Table(table))
}

pub(super) fn select(index: Value, mut args: Vec<Value>) -> Result<Value> {
    let index = match index {
        Value::Number(n) => n.coerce_int()?,
        Value::String(s) => {
            if s == "#" {
                // Shortcut if requesting length
                return Ok(Value::int(args.len() as i64));
            }
            Numeric::from_str(&s)?.coerce_int()?
        }
        _ => panic!("expected number or '#'"),
    };

    Ok(Value::Mult(match index.cmp(&0) {
        Ordering::Greater => {
            let index = index as usize;
            if index >= args.len() {
                vec![]
            } else {
                args.split_off(index - 1)
            }
        }
        Ordering::Less => {
            let index = index.abs() as usize;
            if index >= args.len() {
                panic!("invalid index")
            } else {
                args.split_off(args.len() - index)
            }
        }
        Ordering::Equal => panic!("invalid index"),
    }))
}

pub(super) fn tonumber(v: Value, base: Value) -> Result<Value> {
    let base = match base {
        Value::Nil => None,
        v => Some(v.to_number_coerce()?.coerce_int()? as u32),
    };
    let num = match v {
        Value::Number(n) => {
            if base.is_some() {
                return err!(LuaError::ExpectedType(ValueType::String, ValueType::Number));
            } else {
                Value::Number(n)
            }
        }
        Value::String(s) => match Numeric::from_str_radix(&s, base) {
            Ok(n) => Value::Number(n),
            Err(..) => Value::Nil,
        },
        _ => Value::Nil,
    };
    Ok(num)
}

pub(super) fn tostring(v: Value) -> Result<Value> {
    Ok(Value::String(format!("{}", v)))
}

pub(super) fn r#type(v: Value) -> Result<Value> {
    Ok(Value::String(format!("{}", v.value_type())))
}
