use std::cell::RefCell;
use std::collections::HashSet;
use std::hash::{Hash, Hasher};
use std::io::Write;
use std::mem;
use std::rc::Rc;

use rand::{thread_rng, SeedableRng};
use rand_xoshiro::Xoshiro256StarStar;

mod code;
mod operators;
mod stdlib;
mod value;

use crate::ast::Parser;
use crate::compiler::Compiler;
use crate::error::{LuaError, LuaSrc, LuaTraceScope, Result};

pub use code::{Code, OpCode, Oper};
pub use operators::{BinOp, UnOp};
use value::Numeric;
pub use value::{
    FuncBuiltin, FuncClosure, FuncDef, FuncObject, LocalType, Source, Table, Value, ValueType,
    VarType,
};

#[derive(Clone, Debug)]
pub struct GcObject {
    value: Rc<RefCell<Value>>,
}

impl GcObject {
    fn new() -> Self {
        Self {
            value: Rc::new(RefCell::new(Value::Nil)),
        }
    }
}

impl PartialEq for GcObject {
    fn eq(&self, other: &Self) -> bool {
        Rc::as_ptr(&self.value) == Rc::as_ptr(&other.value)
    }
}

impl Eq for GcObject {}

impl Hash for GcObject {
    fn hash<H: Hasher>(&self, state: &mut H) {
        Rc::as_ptr(&self.value).hash(state);
    }
}

struct Heap {
    heap: HashSet<GcObject>,
}

impl Heap {
    fn new() -> Self {
        Self {
            heap: HashSet::new(),
        }
    }

    fn alloc(&mut self) -> Rc<RefCell<Value>> {
        let var = GcObject::new();
        self.heap.insert(var.clone());
        var.value
    }
}

struct StackFrame {
    ret_pc: (usize, usize),
    ret_reg: Option<usize>,
    trace: (LuaSrc, LuaTraceScope),
    protected: bool,
    regs: Vec<Value>,
    locals: Vec<(Rc<RefCell<Value>>, String)>,
    ups: Vec<(Rc<RefCell<Value>>, String)>,
    varargs: Vec<Value>,
}

pub struct VM<'a> {
    pc: (usize, usize),
    code: Code,
    env: Rc<RefCell<Table>>,
    heap: Heap,
    frames: Vec<StackFrame>,
    rng: Xoshiro256StarStar,
    out: &'a mut dyn Write,
}

impl<'a> VM<'a> {
    pub fn create(out: &'a mut dyn Write) -> Self {
        let mut env = Table::default();
        stdlib::load(&mut env);

        Self {
            pc: (0, 0),
            code: Code::new(),
            env: Rc::new(RefCell::new(env)),
            heap: Heap::new(),
            frames: Vec::new(),
            rng: Xoshiro256StarStar::from_rng(thread_rng()).unwrap(),
            out,
        }
    }

    pub fn load_str(&mut self, input: &str) -> Result<FuncDef> {
        let ast = Parser::parse(input)?;

        let main = Compiler::compile(ast, &mut self.code);

        assert!(main.ups.is_empty());
        assert!(!main.varargs);
        let main = FuncDef::Defined(FuncClosure {
            regs: main.regs,
            locals: main.locals,
            ups: Vec::new(),
            varargs: false,
            chunk: main.chunk,
        });

        Ok(main)
    }

    pub fn run_str(&mut self, input: &str) -> Result<()> {
        let main = self.load_str(input)?;
        self.call(LuaSrc::Rust, Rc::new(main), None, None)?;

        self.run_loop()?;

        assert!(self.frames.is_empty());
        self.out.flush().unwrap();

        Ok(())
    }

    fn call_protected(&mut self, func: Rc<FuncDef>, args: Vec<Value>) -> Result<Value> {
        // register for return value can be anything, should not be used
        match &*func {
            FuncDef::Builtin(func) => {
                // Required to call prepare here because we need to insert protected frame
                self.call_prepare_builtin_protected(LuaSrc::Rust, Some(usize::MAX));
                let callable = func.func;
                let ret = callable(self, args);
                // Popping because there is no return opcode in case of built in function
                self.frames.pop();
                ret
            }
            FuncDef::Defined(func) => {
                self.call_prepare_closure(LuaSrc::Rust, func, args, Some(usize::MAX), true)?;
                self.run_loop()
            }
        }
    }

    fn call(
        &mut self,
        src: LuaSrc,
        func: Rc<FuncDef>,
        reg_args: Option<usize>,
        reg_ret: Option<usize>,
    ) -> Result<()> {
        let args = self.call_args(reg_args);
        match &*func {
            FuncDef::Builtin(func) => {
                // Not inserting a frame,
                // only reason it is needed is because of the error trace,
                // that is instead inserted upon error below
                let callable = func.func;
                let ret = callable(self, args).map_err(move |mut e| {
                    // Insert two trace lines here,
                    // first the trace for the builtin function that it originates from,
                    // and also the trace for the position where the function was called from
                    e = e.trace(LuaSrc::Rust, LuaTraceScope::Function(func.name.to_string()));
                    e.trace(src, LuaTraceScope::Main)
                })?;
                if let Some(reg_ret) = reg_ret {
                    self.frames.last_mut().unwrap().regs[reg_ret] = ret;
                }
            }
            FuncDef::Defined(func) => {
                self.call_prepare_closure(src, func, args, reg_ret, false)?;
            }
        }

        Ok(())
    }

    fn call_args(&mut self, reg: Option<usize>) -> Vec<Value> {
        match reg {
            Some(reg) => {
                let frame = self.frames.last_mut().unwrap();
                mem::replace(&mut frame.regs[reg], Value::Nil).into_vec()
            }
            None => Vec::new(),
        }
    }

    fn call_prepare_builtin_protected(&mut self, src: LuaSrc, ret_reg: Option<usize>) {
        self.frames.push(StackFrame {
            ret_pc: self.pc,
            ret_reg,
            trace: (src, LuaTraceScope::Main),
            protected: true,
            regs: Vec::new(),
            locals: Vec::new(),
            ups: Vec::new(),
            varargs: Vec::new(),
        });
    }

    fn call_prepare_closure(
        &mut self,
        src: LuaSrc,
        func: &FuncClosure,
        args: Vec<Value>,
        ret_reg: Option<usize>,
        protected: bool,
    ) -> Result<()> {
        let mut args = args.into_iter();

        let mut locals = Vec::with_capacity(func.locals.len());
        for (local_type, name) in func.locals.iter() {
            let local = match local_type {
                LocalType::Local => {
                    let var = self.heap.alloc();
                    (var, name.clone())
                }
                LocalType::Param => {
                    let var = self.heap.alloc();
                    if let Some(val) = args.next() {
                        var.replace(val);
                    }
                    (var, name.clone())
                }
            };
            locals.push(local);
        }

        let mut ups = Vec::with_capacity(func.ups.len());
        for (var, name) in func.ups.iter() {
            ups.push((var.clone(), name.clone()));
        }

        self.frames.push(StackFrame {
            ret_pc: self.pc,
            ret_reg,
            trace: (src, LuaTraceScope::Main),
            protected,
            regs: vec![Value::Nil; func.regs],
            locals,
            ups,
            varargs: if func.varargs {
                args.collect()
            } else {
                Vec::new()
            },
        });
        self.pc = (func.chunk, 0);

        Ok(())
    }

    fn jump(&mut self, offset: i64) {
        if offset >= 0 {
            self.pc = (self.pc.0, self.pc.1 + offset as usize);
        } else {
            let offset = offset.abs() as usize;
            assert!(self.pc.1 >= offset);
            self.pc = (self.pc.0, self.pc.1 - offset);
        }
    }

    fn run_loop(&mut self) -> Result<Value> {
        loop {
            match self.run_next() {
                Ok(stop) => {
                    if let Some(ret) = stop {
                        return Ok(ret);
                    }
                }
                Err(mut e) => {
                    // Unroll frames until encountering protected frame, then return,
                    // if no protected frame is found, unroll everything and then return
                    while let Some(frame) = self.frames.pop() {
                        e = e.trace(frame.trace.0, frame.trace.1);
                        if frame.protected {
                            self.pc = frame.ret_pc;
                            break;
                        }
                    }

                    return Err(e);
                }
            }
        }
    }

    fn run_next(&mut self) -> Result<Option<Value>> {
        let current = self.code.get(self.pc);
        self.pc = (self.pc.0, self.pc.1 + 1);

        match current {
            OpCode::Mov { src_reg, dst_reg } => {
                let frame = self.frames.last_mut().unwrap();
                frame.regs.swap(*src_reg, *dst_reg);
            }
            OpCode::MovMult {
                src_reg,
                ind,
                dst_reg,
            } => {
                let frame = self.frames.last_mut().unwrap();
                let val = match &frame.regs[*src_reg] {
                    Value::Mult(vec) => vec.get(*ind).cloned().unwrap_or(Value::Nil),
                    v if *ind == 0 => v.clone(),
                    _ => Value::Nil,
                };
                frame.regs[*dst_reg] = val;
            }
            OpCode::Copy { src_reg, dst_reg } => {
                let frame = self.frames.last_mut().unwrap();
                frame.regs[*dst_reg] = frame.regs[*src_reg].clone();
            }
            OpCode::Lit { val, dst_reg } => {
                let frame = self.frames.last_mut().unwrap();
                frame.regs[*dst_reg] = val.clone();
            }
            OpCode::GlobalGet { name, dst_reg } => {
                let frame = self.frames.last_mut().unwrap();
                let env = self.env.borrow();
                frame.regs[*dst_reg] = env.get(&Value::String(name.to_string()));
            }
            OpCode::GlobalSet { src_reg, name } => {
                let frame = self.frames.last_mut().unwrap();
                let mut env = self.env.borrow_mut();
                let val = mem::replace(&mut frame.regs[*src_reg], Value::Nil).into_single();
                env.set(Value::String(name.clone()), val)?;
            }
            OpCode::LocalGet { src_loc, dst_reg } => {
                let frame = self.frames.last_mut().unwrap();
                let (var, _) = frame.locals.get(*src_loc).unwrap();
                let val = var.borrow().clone();
                frame.regs[*dst_reg] = val;
            }
            OpCode::LocalSet { src_reg, dst_loc } => {
                let frame = self.frames.last_mut().unwrap();
                let (var, _) = frame.locals.get(*dst_loc).unwrap();
                let val = mem::replace(&mut frame.regs[*src_reg], Value::Nil).into_single();
                var.replace(val);
            }
            OpCode::UpGet { src_up, dst_reg } => {
                let frame = self.frames.last_mut().unwrap();
                let (var, _) = frame.ups.get(*src_up).unwrap();
                let val = var.borrow().clone();
                frame.regs[*dst_reg] = val;
            }
            OpCode::UpSet { src_reg, dst_up } => {
                let frame = self.frames.last_mut().unwrap();
                let (var, _) = frame.ups.get(*dst_up).unwrap();
                let val = mem::replace(&mut frame.regs[*src_reg], Value::Nil).into_single();
                var.replace(val);
            }
            OpCode::Varargs { dst_reg } => {
                let frame = self.frames.last_mut().unwrap();
                let varargs = frame.varargs.clone();
                frame.regs[*dst_reg] = Value::Mult(varargs);
            }
            OpCode::BinOp {
                lhs,
                rhs,
                op,
                dst_reg,
            } => {
                let frame = self.frames.last_mut().unwrap();
                let res = fetch_and_call(frame, &self.env, rhs, move |frame, env, rhs| {
                    fetch_and_call(frame, env, lhs, move |_, _, lhs| {
                        operators::bin_op(lhs, *op, rhs)
                    })
                })?;
                let frame = self.frames.last_mut().unwrap();
                frame.regs[*dst_reg] = res;
            }
            OpCode::UnOp { lhs, op, dst_reg } => {
                let frame = self.frames.last_mut().unwrap();
                let res = fetch_and_call(frame, &self.env, lhs, move |_, _, lhs| {
                    operators::un_op(lhs, *op)
                })?;
                frame.regs[*dst_reg] = res;
            }
            OpCode::Single { reg } => {
                let frame = self.frames.last_mut().unwrap();
                let val = mem::replace(&mut frame.regs[*reg], Value::Nil);
                frame.regs[*reg] = val.into_single();
            }
            OpCode::Append { src_reg, dst_reg } => {
                let frame = self.frames.last_mut().unwrap();
                let val = mem::replace(&mut frame.regs[*src_reg], Value::Nil).into_single();
                match &mut frame.regs[*dst_reg] {
                    Value::Mult(vec) => vec.push(val),
                    _ => panic!("invalid value"),
                }
            }
            OpCode::Extend { src_reg, dst_reg } => {
                let frame = self.frames.last_mut().unwrap();
                let val = mem::replace(&mut frame.regs[*src_reg], Value::Nil).into_vec();
                match &mut frame.regs[*dst_reg] {
                    Value::Mult(vec) => vec.extend(val),
                    _ => panic!("invalid value"),
                }
            }
            OpCode::TableGet {
                tbl_reg,
                ind_reg,
                dst_reg,
            } => {
                let frame = self.frames.last_mut().unwrap();
                let ind = mem::replace(&mut frame.regs[*ind_reg], Value::Nil).into_single();
                let tbl = frame.regs[*tbl_reg].to_single();
                let val = match tbl {
                    Value::Table(tbl) => tbl.borrow().get(&ind),
                    Value::String(..) => {
                        match self.env.borrow().get(&Value::String("string".to_string())) {
                            Value::Table(tbl) => tbl.borrow().get(&ind),
                            _ => panic!(),
                        }
                    }
                    _ => panic!("could not find value in table"),
                };
                frame.regs[*dst_reg] = val;
            }
            OpCode::TableSet {
                src_reg,
                tbl_reg,
                ind_reg,
            } => {
                let frame = self.frames.last_mut().unwrap();
                let val = mem::replace(&mut frame.regs[*src_reg], Value::Nil).into_single();
                let tbl = frame.regs[*tbl_reg].to_table()?;
                let ind = mem::replace(&mut frame.regs[*ind_reg], Value::Nil).into_single();
                tbl.borrow_mut().set(ind, val)?;
            }
            OpCode::TableEmpty { dst_reg } => {
                let frame = self.frames.last_mut().unwrap();
                frame.regs[*dst_reg] = Value::Table(Rc::new(RefCell::new(Table::default())));
            }
            OpCode::Closure { func, dst_reg } => {
                let frame = self.frames.last_mut().unwrap();
                let func = *func.clone();

                let mut ups = Vec::new();
                for var in func.ups.into_iter() {
                    let (var, name) = match var {
                        VarType::Local(ind) => frame.locals.get(ind).unwrap(),
                        VarType::Up(ind) => frame.ups.get(ind).unwrap(),
                    };
                    ups.push((var.clone(), name.to_string()));
                }

                let closure = FuncClosure {
                    regs: func.regs,
                    locals: func.locals,
                    ups,
                    varargs: func.varargs,
                    chunk: func.chunk,
                };
                frame.regs[*dst_reg] = Value::Func(Rc::new(FuncDef::Defined(closure)));
            }
            OpCode::Jump { off } => {
                let off = *off;
                self.jump(off);
            }
            OpCode::JumpIf { cmp_reg, off } => {
                let frame = self.frames.last_mut().unwrap();
                if frame.regs[*cmp_reg].is_truthy() {
                    let off = *off;
                    self.jump(off);
                }
            }
            OpCode::JumpIfNot { cmp_reg, off } => {
                let frame = self.frames.last_mut().unwrap();
                if frame.regs[*cmp_reg].is_falsy() {
                    let off = *off;
                    self.jump(off);
                }
            }
            OpCode::Call {
                pos,
                func_reg,
                args_reg,
                ret_reg,
            } => {
                let frame = self.frames.last_mut().unwrap();
                let func = match frame.regs[*func_reg].to_single() {
                    Value::Func(func) => func.clone(),
                    _ => panic!("value not a function"),
                };

                let pos = *pos;
                let reg_args = *args_reg;
                let reg_ret = *ret_reg;
                self.call(
                    LuaSrc::File("".to_string(), pos),
                    func,
                    Some(reg_args),
                    Some(reg_ret),
                )?;
            }
            OpCode::Return { ret_reg } => {
                let ret_val = mem::replace(
                    &mut self.frames.last_mut().unwrap().regs[*ret_reg],
                    Value::Nil,
                );
                let frame = self.frames.pop().unwrap();

                if frame.protected {
                    self.pc = frame.ret_pc;
                    return Ok(Some(ret_val));
                } else if self.frames.is_empty() {
                    return Ok(Some(ret_val));
                }

                self.pc = frame.ret_pc;
                if let Some(ret_reg) = frame.ret_reg {
                    self.frames.last_mut().unwrap().regs[ret_reg] = ret_val;
                }
            }
        }

        Ok(None)
    }
}

fn fetch_and_call<T>(
    frame: &mut StackFrame,
    env: &Rc<RefCell<Table>>,
    oper: &Oper,
    func: T,
) -> Result<Value>
where
    T: FnOnce(&mut StackFrame, &Rc<RefCell<Table>>, &Value) -> Result<Value>,
{
    match oper {
        Oper::Field(reg_tbl, reg_ind) => {
            let tbl = frame.regs[*reg_tbl].to_table()?;
            let ind = frame.regs[*reg_ind].to_single();
            let tbl = tbl.borrow();
            let val = &tbl.get(ind);
            func(frame, env, val)
        }
        Oper::Global(name) => {
            let val = env.borrow().get(&Value::String(name.to_string()));
            func(frame, env, &val)
        }
        Oper::Local(ind) => {
            let (var, _) = frame.locals.get(*ind).unwrap();
            let var = var.clone();
            let val = &*var.borrow();
            func(frame, env, val)
        }
        Oper::Up(ind) => {
            let (var, _) = frame.ups.get(*ind).unwrap();
            let var = var.clone();
            let val = &*var.borrow();
            func(frame, env, val)
        }
        Oper::Reg(reg) => {
            let val = frame.regs[*reg].clone();
            let val = val.to_single();
            func(frame, env, val)
        }
    }
    .map_err(|e| {
        e.map_lua_error(|e| match e {
            LuaError::FloatToInt(orig) => {
                let (num, src) = match oper {
                    Oper::Field(reg_tbl, reg_ind) => {
                        let tbl = frame.regs[*reg_tbl].to_table().unwrap();
                        let ind = frame.regs[*reg_ind].to_single();
                        let tbl = tbl.borrow();
                        let val = &tbl.get(ind);
                        (val.to_number_coerce().ok(), Source::Field(ind.clone()))
                    }
                    Oper::Global(name) => {
                        let val = env.borrow().get(&Value::String(name.to_string()));
                        (
                            val.to_number_coerce().ok(),
                            Source::Global(name.to_string()),
                        )
                    }
                    Oper::Local(ind) => {
                        let (var, name) = frame.locals.get(*ind).unwrap();
                        let var = var.clone();
                        let val = &*var.borrow();
                        (val.to_number_coerce().ok(), Source::Local(name.to_string()))
                    }
                    Oper::Up(ind) => {
                        let (var, name) = frame.ups.get(*ind).unwrap();
                        let var = var.clone();
                        let val = &*var.borrow();
                        (val.to_number_coerce().ok(), Source::Up(name.to_string()))
                    }
                    Oper::Reg(reg) => {
                        let val = frame.regs[*reg].clone();
                        let val = val.to_single();
                        (val.to_number_coerce().ok(), Source::None)
                    }
                };
                LuaError::FloatToInt(
                    num.and_then(|num| num.coerce_int().err().map(|_| src))
                        .unwrap_or(orig),
                )
            }
            e => e,
        })
    })
}
