use std::fmt;

use crate::token::Pos;
use crate::vm::{BinOp, FuncObject, UnOp, Value};

#[derive(Clone, Debug)]
pub enum Oper {
    Field(usize, usize), // reg_tbl, reg_index
    Global(String),
    Local(usize),
    Up(usize),
    Reg(usize),
}

#[derive(Clone, Debug)]
pub enum OpCode {
    // Register stuff
    Mov {
        src_reg: usize,
        dst_reg: usize,
    },
    MovMult {
        src_reg: usize,
        ind: usize,
        dst_reg: usize,
    },
    Copy {
        src_reg: usize,
        dst_reg: usize,
    },
    Lit {
        val: Value,
        dst_reg: usize,
    },
    GlobalGet {
        name: String,
        dst_reg: usize,
    },
    GlobalSet {
        src_reg: usize,
        name: String,
    },
    LocalGet {
        src_loc: usize,
        dst_reg: usize,
    },
    LocalSet {
        src_reg: usize,
        dst_loc: usize,
    },
    UpGet {
        src_up: usize,
        dst_reg: usize,
    },
    UpSet {
        src_reg: usize,
        dst_up: usize,
    },
    Varargs {
        dst_reg: usize,
    },
    // Operations
    BinOp {
        lhs: Oper,
        rhs: Oper,
        op: BinOp,
        dst_reg: usize,
    },
    UnOp {
        lhs: Oper,
        op: UnOp,
        dst_reg: usize,
    },
    Single {
        reg: usize,
    },
    Append {
        src_reg: usize,
        dst_reg: usize,
    },
    Extend {
        src_reg: usize,
        dst_reg: usize,
    },
    TableGet {
        tbl_reg: usize,
        ind_reg: usize,
        dst_reg: usize,
    },
    TableSet {
        src_reg: usize,
        tbl_reg: usize,
        ind_reg: usize,
    },
    TableEmpty {
        dst_reg: usize,
    },
    Closure {
        func: Box<FuncObject>,
        dst_reg: usize,
    },
    // Jumps (relative)
    Jump {
        off: i64,
    },
    JumpIf {
        cmp_reg: usize,
        off: i64,
    },
    JumpIfNot {
        cmp_reg: usize,
        off: i64,
    },
    // Call stack
    Call {
        pos: Pos,
        func_reg: usize,
        args_reg: usize,
        ret_reg: usize,
    },
    Return {
        ret_reg: usize,
    },
}

pub struct Code {
    chunks: Vec<Vec<OpCode>>,
    current: usize,
}

impl Code {
    pub fn new() -> Self {
        Self {
            chunks: Vec::new(),
            current: 0,
        }
    }

    pub fn get(&self, pos: (usize, usize)) -> &OpCode {
        &self.chunks[pos.0][pos.1]
    }

    pub fn set(&mut self, pos: (usize, usize), op: OpCode) {
        self.chunks[pos.0][pos.1] = op;
    }

    pub fn pos(&self) -> (usize, usize) {
        (self.current, self.chunks[self.current].len())
    }

    pub fn emit(&mut self, op: OpCode) {
        self.chunks[self.current].push(op);
    }

    pub fn emit_all<I: IntoIterator<Item = OpCode>>(&mut self, code: I) {
        self.chunks[self.current].extend(code);
    }

    pub fn new_chunk(&mut self) -> (usize, usize) {
        let label = self.chunks.len();
        let current = self.current;

        self.chunks.push(Vec::new());
        self.current = label;

        (label, current)
    }

    pub fn set_current(&mut self, current: usize) {
        self.current = current;
    }
}

const CODE_WIDTH: usize = 11;

impl fmt::Display for Code {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let mut funcs: Vec<Option<&Box<FuncObject>>> = vec![None; self.chunks.len()];
        let chunk_digits = (self.chunks.len() as f64).log10().ceil() as usize;
        for (i, chunk) in self.chunks.iter().enumerate() {
            if let Some(func) = funcs.get(i).and_then(|o| *o) {
                writeln!(f, "{}", func)?;
            }
            let op_digits = (chunk.len() as f64).log10().ceil() as usize;
            for (j, op) in chunk.iter().enumerate() {
                write!(f, "{:02$}:{:03$}: ", i, j, chunk_digits, op_digits)?;
                let (name, debug) = match op {
                    OpCode::Mov { src_reg, dst_reg } => {
                        ("Mov", format!("r[{}] = r[{}]", dst_reg, src_reg))
                    }
                    OpCode::MovMult {
                        src_reg,
                        ind,
                        dst_reg,
                    } => (
                        "MovMult",
                        format!("r[{}] = r[{}][{}]", dst_reg, src_reg, ind),
                    ),
                    OpCode::Copy { src_reg, dst_reg } => {
                        ("Copy", format!("r[{}] = r[{}]", dst_reg, src_reg))
                    }
                    OpCode::Lit { val, dst_reg } => ("Lit", format!("r[{}] = {:?}", dst_reg, val)),
                    OpCode::GlobalGet { name, dst_reg } => {
                        ("GlobalGet", format!("r[{}] = g[\"{}\"]", dst_reg, name))
                    }
                    OpCode::GlobalSet { src_reg, name } => {
                        ("GlobalSet", format!("g[\"{}\"] = r[{}]", name, src_reg))
                    }
                    OpCode::LocalGet { src_loc, dst_reg } => {
                        ("LocalGet", format!("r[{}] = l[{}]", dst_reg, src_loc))
                    }
                    OpCode::LocalSet { src_reg, dst_loc } => {
                        ("LocalSet", format!("l[{}] = r[{}]", dst_loc, src_reg))
                    }
                    OpCode::UpGet { src_up, dst_reg } => {
                        ("UpGet", format!("r[{}] = u[{}]", dst_reg, src_up))
                    }
                    OpCode::UpSet { src_reg, dst_up } => {
                        ("UpSet", format!("u[{}] = r[{}]", dst_up, src_reg))
                    }
                    OpCode::Varargs { dst_reg } => ("Varargs", format!("r[{}] = ...", dst_reg)),
                    OpCode::BinOp {
                        lhs,
                        rhs,
                        op,
                        dst_reg,
                    } => (
                        "BinOp",
                        format!("r[{}] = {} {:?} {}", dst_reg, lhs, op, rhs),
                    ),
                    OpCode::UnOp { lhs, op, dst_reg } => {
                        ("UnOp", format!("r[{}] = {:?} {}", dst_reg, op, lhs))
                    }
                    OpCode::Single { reg } => ("Single", format!("r[{}] = r[{}][..][0]", reg, reg)),
                    OpCode::Append { src_reg, dst_reg } => (
                        "Append",
                        format!("r[{}] = [r[{}][..], r[{}]]", dst_reg, dst_reg, src_reg),
                    ),
                    OpCode::Extend { src_reg, dst_reg } => (
                        "Extend",
                        format!("r[{}] = [r[{}][..], r[{}][..]]", dst_reg, dst_reg, src_reg),
                    ),
                    OpCode::TableGet {
                        tbl_reg,
                        ind_reg,
                        dst_reg,
                    } => (
                        "TableGet",
                        format!("r[{}] = r[{}][r[{}]]", dst_reg, tbl_reg, ind_reg),
                    ),
                    OpCode::TableSet {
                        src_reg,
                        tbl_reg,
                        ind_reg,
                    } => (
                        "TableSet",
                        format!("r[{}][r[{}]] = r[{}]", tbl_reg, ind_reg, src_reg),
                    ),
                    OpCode::TableEmpty { dst_reg } => {
                        ("TableEmpty", format!("r[{}] = {{}}", dst_reg))
                    }
                    OpCode::Closure { func, dst_reg } => {
                        if func.chunk < funcs.len() {
                            funcs[func.chunk] = Some(func);
                        }
                        (
                            "Closure",
                            format!("r[{}] = close fn_{}", dst_reg, func.chunk),
                        )
                    }
                    OpCode::Jump { off } => (
                        "Jump",
                        format!("pc = {}:{} ({:+})", i, (j as i64) + 1 + off, off),
                    ),
                    OpCode::JumpIf { cmp_reg, off } => (
                        "JumpIf",
                        format!(
                            "if r[{}] then pc = {}:{} ({:+})",
                            cmp_reg,
                            i,
                            (j as i64) + 1 + off,
                            off
                        ),
                    ),
                    OpCode::JumpIfNot { cmp_reg, off } => (
                        "JumpIfNot",
                        format!(
                            "if not r[{}] then pc = {}:{} ({:+})",
                            cmp_reg,
                            i,
                            (j as i64) + 1 + off,
                            off
                        ),
                    ),
                    OpCode::Call {
                        func_reg,
                        args_reg,
                        ret_reg,
                        ..
                    } => (
                        "Call",
                        format!("r[{}] = r[{}](r[{}][..])", ret_reg, func_reg, args_reg),
                    ),
                    OpCode::Return { ret_reg } => ("Return", format!("return r[{}]", ret_reg)),
                };
                writeln!(f, "{:>width$} | {}", name, debug, width = CODE_WIDTH)?;
            }

            writeln!(f)?;
        }

        Ok(())
    }
}

impl fmt::Display for Oper {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Oper::Field(tbl_reg, ind_reg) => write!(f, "r[{}][r[{}]]", tbl_reg, ind_reg),
            Oper::Global(name) => write!(f, "g[\"{}\"]", name),
            Oper::Local(loc) => write!(f, "l[{}]", loc),
            Oper::Up(up) => write!(f, "u[{}]", up),
            Oper::Reg(reg) => write!(f, "r[{}]", reg),
        }
    }
}

impl fmt::Display for FuncObject {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        writeln!(f, "fn_{}():", self.chunk)?;
        let loc_width = (self.locals.len() as f64).log10().ceil() as usize;
        for (i, (typ, name)) in self.locals.iter().enumerate() {
            writeln!(f, "\tloc {:03$}: {:?} {}", i, typ, name, loc_width)?;
        }
        let up_width = (self.ups.len() as f64).log10().ceil() as usize;
        for (i, typ) in self.ups.iter().enumerate() {
            writeln!(f, "\tup {:02$}: {:?}", i, typ, up_width)?;
        }
        writeln!(f, "\tregisters: {}", self.regs)?;
        write!(f, "\tvarargs: {}", self.varargs)
    }
}
