use crate::error::{LuaError, Result};

pub fn from_str_radix_wrapping(str: &str, radix: u32) -> Option<i64> {
    assert!((2..=36).contains(&radix), "radix must be in range [2, 36]");
    let str = str.as_bytes();
    let mut out: i64 = 0;

    let (positive, digits) = match str[0] {
        b'-' => (false, &str[1..]),
        b'+' => (true, &str[1..]),
        _ => (true, str),
    };
    let digits = if digits.len() >= 2 && matches!(&digits[0..2], b"0x" | b"0X") {
        &digits[2..]
    } else {
        digits
    };
    if digits.is_empty() {
        return None;
    }

    for &d in digits {
        let d = match (d as char).to_digit(radix) {
            Some(d) => d,
            None => return None,
        };
        out = out.wrapping_mul(radix as i64);
        out = if positive {
            out.wrapping_add(d as i64)
        } else {
            out.wrapping_sub(d as i64)
        };
    }

    Some(out)
}

// Inf and NaN are invalid numeric literals in Lua, thus filter them out
pub fn parse_f64_dec(str: &str) -> Result<f64> {
    let f: f64 = str.parse()?;
    if f.is_nan() || f.is_infinite() {
        err!(LuaError::Custom(
            "inf and nan are invalid numeric literals".to_string()
        ))
    } else {
        Ok(f)
    }
}

// maximum number of significant digits to read
// (to avoid overflows even with single floats)
const MAX_SIG_DIG: usize = 30;

// Convert a string containing a hexadecimal float literal into f64,
// this code is derived from the hexadecimal float parsing code of the Lua project compiler
pub fn parse_f64_hex(str: &str) -> Option<f64> {
    let str = str.trim().as_bytes();

    let mut acc: f64 = 0.0; // accumulator/result
    let mut sig_dig = 0; // significant digits
    let mut non_sig_dig = 0; // non-significant digits
    let mut exp: i64 = 0; // exponent correction
    let mut has_dot = false;

    let (str, neg) = match str[0] {
        b'-' => (&str[1..], true),
        b'+' => (&str[1..], false),
        _ => (str, false),
    };
    // Must contain 0x
    if !matches!(&str[..2], b"0x" | b"0X") {
        return None;
    }

    // Read numeral part
    let mut str = str[2..].iter().peekable();
    while let Some(b) = str.peek() {
        if matches!(b, b'.') {
            if has_dot {
                return None;
            }
            has_dot = true;
        } else if let Some(d) = (**b as char).to_digit(16) {
            if sig_dig == 0 && d == 0 {
                // No significant digits yet and current is neither
                non_sig_dig += 1;
            } else if sig_dig < MAX_SIG_DIG {
                // Can be read without overflow
                sig_dig += 1;
                acc = acc.mul_add(16.0, if neg { -(d as f64) } else { d as f64 });
            } else {
                // Too many digits, but still correct exponent
                exp += 4; // Each digit increments exponent by 4
            }
            if has_dot {
                // If decimal digit, correct exponent
                exp -= 4; // Each digit decrements exponent by 4
            }
        } else {
            break;
        }
        str.next();
    }
    // Check if read anything
    if non_sig_dig + sig_dig == 0 {
        return None;
    }

    // Read exponent
    if matches!(str.peek(), Some(b'p' | b'P')) {
        str.next();
        let mut exp_lit: i64 = 0;

        let exp_neg = match str.peek() {
            Some(b'-') => {
                str.next();
                true
            }
            Some(b'+') => {
                str.next();
                false
            }
            _ => false,
        };
        // Must have at least one digit
        str.peek()?;

        for b in &mut str {
            match (*b as char).to_digit(10) {
                Some(d) => {
                    exp_lit *= 10;
                    if exp_neg {
                        exp_lit -= d as i64;
                    } else {
                        exp_lit += d as i64;
                    }
                }
                None => return None,
            }
        }

        exp += exp_lit;
    }

    // Consumed everything, but still stuff left, invalid
    if str.next().is_some() {
        return None;
    }

    Some(acc * (exp as f64).exp2())
}
