use std::fmt;

use thiserror::Error;

use crate::token::{Pos, Token};
use crate::vm::{Source, ValueType};

pub type Result<T> = std::result::Result<T, Error>;

macro_rules! err {
    ( $pos:expr, $typ:expr ) => {
        Err(crate::error::Error::Lua {
            typ: $typ,
            pos: crate::error::LuaPos(Some($pos)),
            trace: Default::default(),
        })
    };
    ( $typ:expr ) => {
        Err(crate::error::Error::Lua {
            typ: $typ,
            pos: crate::error::LuaPos(None),
            trace: Default::default(),
        })
    };
}

#[derive(Debug, Error)]
pub enum Error {
    #[error("format error: {0}")]
    Fmt(#[from] std::fmt::Error),
    #[error("IO error: {0}")]
    Io(#[from] std::io::Error),
    #[error("parse int: {0}")]
    ParseInt(#[from] std::num::ParseIntError),
    #[error("parse float: {0}")]
    ParseFloat(#[from] std::num::ParseFloatError),
    #[error("{pos}{typ}{trace}")]
    Lua {
        typ: LuaError,
        pos: LuaPos,
        trace: LuaTraces,
    },
}

#[derive(Debug, Error)]
pub enum LuaError {
    #[error("wrong number of arguments")]
    ArgumentCount,
    #[error("assertion failed!")]
    AssertFailed,
    #[error("{0}")]
    Custom(String),
    #[error("attempt to divide by zero")]
    DivideByZero,
    #[error("interval is empty")]
    EmptyInterval,
    #[error("{0} expected, got {0}")]
    ExpectedType(ValueType, ValueType),
    #[error("value expected")]
    ExpectedValue,
    #[error("number{0} has no integer representation")]
    FloatToInt(Source),
    #[error("invalid table index")]
    TableIndex,
    #[error("unexpected end of file")]
    UnexpectedEof,
    #[error("unexpected token '{0:?}'")]
    UnexpectedToken(Token),
}

#[derive(Debug)]
pub struct LuaPos(pub Option<Pos>);

#[derive(Debug, Default)]
pub struct LuaTraces(Vec<LuaTrace>);

#[derive(Debug)]
pub struct LuaTrace {
    src: LuaSrc,
    scope: LuaTraceScope,
}

#[derive(Debug)]
pub enum LuaSrc {
    Rust,
    File(String, Pos),
    Str(String, Pos),
}

#[derive(Debug)]
pub enum LuaTraceScope {
    Main,
    Function(String),
}

impl Error {
    pub fn map_lua_error<F>(self, func: F) -> Self
    where
        F: FnOnce(LuaError) -> LuaError,
    {
        match self {
            Error::Lua { typ, pos, trace } => Error::Lua {
                typ: func(typ),
                pos,
                trace,
            },
            e => e,
        }
    }

    pub fn source(self, source: Source) -> Self {
        match self {
            Error::Lua { typ, pos, trace } => Error::Lua {
                typ: typ.source(source),
                pos,
                trace,
            },
            e => e,
        }
    }

    pub fn trace(self, src: LuaSrc, scope: LuaTraceScope) -> Self {
        match self {
            Error::Lua {
                typ,
                pos,
                mut trace,
            } => {
                trace.0.push(LuaTrace { src, scope });
                Error::Lua { typ, pos, trace }
            }
            e => e,
        }
    }
}

impl LuaError {
    fn source(self, source: Source) -> Self {
        match self {
            LuaError::FloatToInt(..) => LuaError::FloatToInt(source),
            e => e,
        }
    }
}

impl fmt::Display for LuaPos {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        if let Some(pos) = self.0 {
            write!(f, "{}:{}:", pos.line() + 1, pos.col() + 1)?;
        }
        Ok(())
    }
}

impl fmt::Display for LuaTraces {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        for trace in self.0.iter() {
            write!(f, "\n\t")?;
            match &trace.src {
                LuaSrc::Rust => write!(f, "[Rust]:")?,
                LuaSrc::File(filename, pos) => {
                    write!(f, "{}:{}:{}", filename, pos.line() + 1, pos.col() + 1)?
                }
                LuaSrc::Str(str, pos) => {
                    write!(f, "[string \"")?;
                    let mut lines = str.lines();
                    if let Some(line) = lines.next() {
                        write!(f, "{}", line)?;
                        if lines.next().is_some() {
                            write!(f, "...")?;
                        }
                    }
                    write!(f, "\"]:{}:{}:", pos.line() + 1, pos.col() + 1)?;
                }
            }
            match &trace.scope {
                LuaTraceScope::Main => write!(f, " in main chunk")?,
                LuaTraceScope::Function(func) => write!(f, " in function '{}'", func)?,
            }
        }
        Ok(())
    }
}
