use crate::ast::Stat;
use crate::compiler::{Compiler, ScopeType};
use crate::vm::{OpCode, VarType};

impl<'a> Compiler<'a> {
    pub(super) fn compile_stat(&mut self, stat: Stat) {
        match stat {
            Stat::Nop => {}
            Stat::Ass(ass) => self.compile_ass(ass),
            Stat::Call(pos, exp, args) => {
                let reg_ret = self.compile_call(pos, exp, args);
                self.scopes.reg_free(reg_ret);
            }
            Stat::CallMethod(pos, prefix, name, args) => {
                let reg_ret = self.compile_call_method(pos, *prefix, name, args);
                self.scopes.reg_free(reg_ret);
            }
            Stat::Func(name, func) => {
                let src_reg = self.compile_func(func);
                match self.scopes.get(&name) {
                    Some((var, _)) => self.code.emit(match var {
                        VarType::Local(dst_loc) => OpCode::LocalSet { src_reg, dst_loc },
                        VarType::Up(dst_up) => OpCode::UpSet { src_reg, dst_up },
                    }),
                    None => self.code.emit(OpCode::GlobalSet { src_reg, name }),
                }
                self.scopes.reg_free(src_reg);
            }
            Stat::LocalFunc(name, func) => {
                let src_reg = self.compile_func(func);
                let dst_loc = self.scopes.declare_local(name, None);
                self.code.emit(OpCode::LocalSet { src_reg, dst_loc });
                self.scopes.reg_free(src_reg);
            }
            Stat::Local(local) => self.compile_local(local),
            Stat::Do(block) => {
                self.scope_enter(ScopeType::Do);

                self.compile_block(block);

                self.scope_leave(ScopeType::Do);
            }
            Stat::If(iff) => {
                self.compile_if(iff);
            }
            Stat::Break => {
                // Jump to end of loop, placeholder
                let pos = self.code.pos();
                self.code.emit(OpCode::Jump { off: 0 });
                self.scopes.break_insert(pos);
            }
            Stat::While(r#while) => {
                self.compile_while(r#while);
            }
            Stat::Repeat(repeat) => {
                self.compile_repeat(repeat);
            }
            Stat::For(r#for) => {
                self.compile_for(r#for);
            }
            Stat::Goto(label) => {
                self.scopes.goto_insert(label, self.code.pos());
                self.code.emit(OpCode::Jump { off: 0 }); // Placeholder
            }
            Stat::Label(label) => {
                self.scopes.label_insert(label, self.code.pos());
            }
        }
    }
}
