use std::collections::HashMap;

mod assignment;
mod block;
mod chunk;
mod expression;
mod r#for;
mod function;
mod r#if;
mod local;
mod prefix_exp;
mod repeat;
mod statement;
mod r#while;

use crate::ast::{Attr, Chunk};
use crate::bitset::BitSet;
use crate::vm::{Code, FuncObject, LocalType, OpCode, VarType};

pub struct Compiler<'a> {
    // Tracking local variables
    scopes: Scopes,
    code: &'a mut Code,
}

impl<'a> Compiler<'a> {
    pub fn compile(chunk: Chunk, code: &'a mut Code) -> FuncObject {
        let mut compiler = Self {
            scopes: Scopes::new(),
            code,
        };

        compiler.compile_chunk(chunk)
    }

    fn scope_enter(&mut self, typ: ScopeType) {
        self.scopes.open(typ);
    }

    fn scope_leave(&mut self, typ: ScopeType) -> Option<Scope> {
        let (scope, sub) = self.scopes.close(typ);

        // Replace break placeholders
        for break_pos in sub.breaks.into_iter() {
            let off = offset(break_pos, self.code.pos());
            self.code.set(break_pos, OpCode::Jump { off });
        }

        // Replace goto placeholders, as far as can be resolved
        for (label, goto_pos) in sub.gotos.into_iter() {
            match sub.labels.get(&label) {
                Some(label_pos) => {
                    let off = offset(goto_pos, *label_pos);
                    self.code.set(goto_pos, OpCode::Jump { off });
                }
                // Bubble to parent scope, except if function scope was closed
                None => {
                    if scope.is_some() {
                        panic!("no label found in scope for goto")
                    } else {
                        self.scopes.goto_insert(label, goto_pos);
                    }
                }
            };
        }

        scope
    }
}

struct Scopes {
    scopes: Vec<Scope>,
}

impl Scopes {
    fn new() -> Self {
        Self { scopes: Vec::new() }
    }

    fn open(&mut self, typ: ScopeType) {
        match typ {
            ScopeType::Func => self.scopes.push(Scope::new()),
            t => self.scopes.last_mut().unwrap().open(t),
        }
    }

    fn close(&mut self, typ: ScopeType) -> (Option<Scope>, SubScope) {
        let closed_sub = self.scopes.last_mut().unwrap().close();
        assert!(closed_sub.typ == typ);
        let closed_func = match closed_sub.typ {
            ScopeType::Func => {
                let closed_func = self.scopes.pop().unwrap();
                assert!(closed_func.regs.is_empty());
                assert!(closed_func.sub_scopes.is_empty());
                Some(closed_func)
            }
            _ => None,
        };
        (closed_func, closed_sub)
    }

    fn reg_reserve(&mut self) -> usize {
        self.scopes.last_mut().unwrap().regs.reserve()
    }

    fn reg_free(&mut self, ind: usize) {
        self.scopes.last_mut().unwrap().regs.free(ind);
    }

    fn get(&mut self, name: &str) -> Option<(VarType, Option<Attr>)> {
        // Find variable
        for (depth, scope) in self.scopes.iter().rev().enumerate() {
            if let Some(var) = scope.get_var(name) {
                let attr = scope.get_attr(var);
                if depth == 0 {
                    // Local found in current scope, just return
                    return Some((*var, attr));
                } else {
                    // Insert upvalues in all scopes above where local was declared
                    let len = self.scopes.len();
                    let mut upper = *var;
                    for scope in self.scopes.iter_mut().skip(len - depth) {
                        let up = scope.declare_up(name.to_string(), upper, attr);
                        upper = VarType::Up(up);
                    }
                    return Some((upper, attr));
                }
            }
        }
        None
    }

    fn declare_local(&mut self, name: String, attr: Option<Attr>) -> usize {
        self.scopes
            .last_mut()
            .unwrap()
            .declare_local(name, LocalType::Local, attr)
    }

    fn declare_params(&mut self, params: Vec<String>) {
        let scope = self.scopes.last_mut().unwrap();
        for param in params.into_iter() {
            scope.declare_local(param, LocalType::Param, None);
        }
    }

    fn break_insert(&mut self, pos: (usize, usize)) {
        self.scopes.last_mut().unwrap().break_insert(pos);
    }

    fn goto_insert(&mut self, name: String, pos: (usize, usize)) {
        let scope = self.scopes.last_mut().unwrap();
        scope.sub_scopes.last_mut().unwrap().gotos.push((name, pos));
    }

    fn label_insert(&mut self, name: String, pos: (usize, usize)) {
        let scope = self.scopes.last_mut().unwrap();
        scope
            .sub_scopes
            .last_mut()
            .unwrap()
            .labels
            .insert(name, pos);
    }
}

#[derive(PartialEq)]
enum ScopeType {
    Func,
    Loop,
    Do,
}

pub struct Scope {
    sub_scopes: Vec<SubScope>,
    // Registers
    regs: BitSet,
    // Variable slots
    locals: Vec<(LocalType, String, Option<Attr>)>,
    ups: Vec<(VarType, Option<Attr>)>,
}

struct SubScope {
    typ: ScopeType,
    // Variable search dict
    dict: HashMap<String, VarType>,
    // Breaks
    breaks: Vec<(usize, usize)>,
    // Gotos
    gotos: Vec<(String, (usize, usize))>,
    labels: HashMap<String, (usize, usize)>,
}

impl Scope {
    fn new() -> Self {
        Self {
            regs: BitSet::new(),
            sub_scopes: vec![SubScope {
                typ: ScopeType::Func,
                dict: HashMap::new(),
                breaks: Vec::new(),
                gotos: Vec::new(),
                labels: HashMap::new(),
            }],
            locals: Vec::new(),
            ups: Vec::new(),
        }
    }

    fn open(&mut self, typ: ScopeType) {
        assert!(!matches!(typ, ScopeType::Func));
        self.sub_scopes.push(SubScope {
            typ,
            dict: HashMap::new(),
            breaks: Vec::new(),
            gotos: Vec::new(),
            labels: HashMap::new(),
        });
    }

    fn close(&mut self) -> SubScope {
        self.sub_scopes.pop().unwrap()
    }

    fn get_var(&self, name: &str) -> Option<&VarType> {
        for sub in self.sub_scopes.iter().rev() {
            if let Some(var) = sub.dict.get(name) {
                return Some(var);
            }
        }
        None
    }

    fn get_attr(&self, var: &VarType) -> Option<Attr> {
        match var {
            VarType::Local(ind) => self.locals[*ind].2,
            VarType::Up(ind) => self.ups[*ind].1,
        }
    }

    fn declare_local(&mut self, name: String, typ: LocalType, attr: Option<Attr>) -> usize {
        let ind = self.locals.len();
        self.locals.push((typ, name.clone(), attr));
        self.sub_scopes
            .last_mut()
            .unwrap()
            .dict
            .insert(name, VarType::Local(ind));
        ind
    }

    fn declare_up(&mut self, name: String, upper: VarType, attr: Option<Attr>) -> usize {
        let ind = self.ups.len();
        self.ups.push((upper, attr));
        // Always insert upvalue on bottom of sub-scope stack,
        // the same name will always resolve to that upvalue,
        // regardless of how many sub-scopes we have in this function,
        // if we would insert on top of stack then when that sub-scope
        // is dropped we would have to resolve and insert the same upvalue again
        self.sub_scopes
            .first_mut()
            .unwrap()
            .dict
            .insert(name, VarType::Up(ind));
        ind
    }

    fn break_insert(&mut self, pos: (usize, usize)) {
        for sub in self.sub_scopes.iter_mut().rev() {
            if let ScopeType::Loop = sub.typ {
                sub.breaks.push(pos);
                return;
            }
        }
        panic!("not in loop")
    }
}

/// Expects position of jump and position to jump to
fn offset(jump: (usize, usize), target: (usize, usize)) -> i64 {
    assert_eq!(jump.0, target.0);
    // Compiler::pos() gets the position of the next emitted opcode,
    // usually one would save the result from pos(), emit a placeholder,
    // and then later use the saved pos to fill placeholder position,
    // this means that we need to adjust by one here
    target.1 as i64 - jump.1 as i64 - 1
}
