use crate::ast::{Exp, Func, PrefixExp};
use crate::compiler::{Compiler, ScopeType};
use crate::token::Pos;
use crate::vm::{FuncObject, OpCode, Value};

impl<'a> Compiler<'a> {
    pub(super) fn compile_func(&mut self, func: Func) -> usize {
        self.scope_enter(ScopeType::Func);
        self.scopes.declare_params(func.params);

        // Compile into new chunk
        let (label, current_save) = self.code.new_chunk();
        if !self.compile_block(*func.block) {
            let ret_reg = self.scopes.reg_reserve();
            self.code.emit(OpCode::Lit {
                val: Value::empty(),
                dst_reg: ret_reg,
            });
            self.code.emit(OpCode::Return { ret_reg }); // Ensure that function has at least one return
            self.scopes.reg_free(ret_reg);
        }
        self.code.set_current(current_save);

        let scope = self.scope_leave(ScopeType::Func).unwrap();
        let def = FuncObject {
            regs: scope.regs.count(),
            varargs: func.varargs,
            locals: scope.locals.into_iter().map(|(t, s, _)| (t, s)).collect(),
            ups: scope.ups.into_iter().map(|(t, _)| t).collect(),
            chunk: label,
        };

        let dst_reg = self.scopes.reg_reserve();
        self.code.emit(OpCode::Closure {
            func: Box::new(def),
            dst_reg,
        });
        dst_reg
    }

    pub(super) fn compile_call(
        &mut self,
        pos: Pos,
        prefix: PrefixExp,
        mut args: Vec<Exp>,
    ) -> usize {
        let args_reg = self.scopes.reg_reserve();
        self.code.emit(OpCode::Lit {
            val: Value::empty_cap(args.len()),
            dst_reg: args_reg,
        });
        let last_arg = args.pop();
        for arg in args.into_iter() {
            let arg_reg = self.compile_exp(arg);
            self.code.emit(OpCode::Append {
                src_reg: arg_reg,
                dst_reg: args_reg,
            });
            self.scopes.reg_free(arg_reg);
        }
        if let Some(arg) = last_arg {
            let arg_reg = self.compile_exp(arg);
            self.code.emit(OpCode::Extend {
                src_reg: arg_reg,
                dst_reg: args_reg,
            });
            self.scopes.reg_free(arg_reg);
        }

        let func_reg = self.compile_prefix_exp(prefix);

        let ret_reg = self.scopes.reg_reserve();
        self.code.emit(OpCode::Call {
            pos,
            func_reg,
            args_reg,
            ret_reg,
        });

        self.scopes.reg_free(args_reg);
        self.scopes.reg_free(func_reg);

        ret_reg
    }

    pub(super) fn compile_call_method(
        &mut self,
        pos: Pos,
        prefix: PrefixExp,
        name: String,
        mut args: Vec<Exp>,
    ) -> usize {
        let obj_reg = self.compile_prefix_exp(prefix);
        let func_reg = self.scopes.reg_reserve();
        self.code.emit(OpCode::Copy {
            src_reg: obj_reg,
            dst_reg: func_reg,
        });

        let args_reg = self.scopes.reg_reserve();
        self.code.emit(OpCode::Lit {
            val: Value::empty(),
            dst_reg: args_reg,
        });
        self.code.emit(OpCode::Append {
            src_reg: func_reg,
            dst_reg: args_reg,
        });
        let last_arg = args.pop();
        for arg in args.into_iter() {
            let arg_reg = self.compile_exp(arg);
            self.code.emit(OpCode::Append {
                src_reg: arg_reg,
                dst_reg: args_reg,
            });
            self.scopes.reg_free(arg_reg);
        }
        if let Some(arg) = last_arg {
            let arg_reg = self.compile_exp(arg);
            self.code.emit(OpCode::Extend {
                src_reg: arg_reg,
                dst_reg: args_reg,
            });
            self.scopes.reg_free(arg_reg);
        }

        self.code.emit(OpCode::Lit {
            val: Value::String(name),
            dst_reg: func_reg,
        });
        self.code.emit(OpCode::TableGet {
            tbl_reg: obj_reg,
            ind_reg: func_reg,
            dst_reg: func_reg,
        });

        let ret_reg = self.scopes.reg_reserve();
        self.code.emit(OpCode::Call {
            pos,
            func_reg,
            args_reg,
            ret_reg,
        });

        self.scopes.reg_free(obj_reg);
        self.scopes.reg_free(args_reg);
        self.scopes.reg_free(func_reg);

        ret_reg
    }
}
