use crate::ast::Chunk;
use crate::compiler::{Compiler, ScopeType};
use crate::vm::{FuncObject, LocalType, OpCode, Value};

impl<'a> Compiler<'a> {
    pub(super) fn compile_chunk(&mut self, chunk: Chunk) -> FuncObject {
        let (label, _) = self.code.new_chunk();
        self.scope_enter(ScopeType::Func);

        if !self.compile_block(chunk.block) {
            let ret_reg = self.scopes.reg_reserve();
            self.code.emit(OpCode::Lit {
                val: Value::empty(),
                dst_reg: ret_reg,
            });
            self.code.emit(OpCode::Return { ret_reg }); // Ensure that function has at least one return
            self.scopes.reg_free(ret_reg);
        }

        let scope = self.scope_leave(ScopeType::Func).unwrap();
        assert!(scope.ups.is_empty());
        FuncObject {
            regs: scope.regs.count(),
            locals: scope
                .locals
                .into_iter()
                .map(|(t, s, _)| {
                    assert!(matches!(t, LocalType::Local));
                    (t, s)
                })
                .collect(),
            ups: Vec::new(),
            varargs: false,
            chunk: label,
        }
    }
}
