use crate::ast::{Exp, Parser};
use crate::error::{LuaError, Result};
use crate::token::{Spanned, Token};
use crate::vm::Value;

/// ```text
/// tableconstructor ::= ‘{’ [fieldlist] ‘}’
/// fieldlist ::= field {fieldsep field} [fieldsep]
/// field ::= ‘[’ exp ‘]’ ‘=’ exp | Name ‘=’ exp | exp
/// fieldsep ::= ‘,’ | ‘;’
/// ```
pub enum TableField {
    Index(Exp, Exp),
    List(Exp),
}

impl<'a> Parser<'a> {
    pub(super) fn parse_table_constructor(&mut self) -> Result<Vec<TableField>> {
        let mut fields = Vec::new();

        expect_token!(self.tokens.next(), Token::CurlO);

        // Consume first field (if there is any)
        if !matches!(self.tokens.peek().map(Spanned::inner), Some(Token::CurlC)) {
            fields.push(self.parse_table_field()?);
        }
        loop {
            match not_eof!(self.tokens.next()).into_tuple() {
                (_, Token::CurlC) => break,
                (_, Token::Comma | Token::Semi) => fields.push(self.parse_table_field()?),
                (pos, tok) => return err!(pos, LuaError::UnexpectedToken(tok)),
            }
        }

        Ok(fields)
    }

    fn parse_table_field(&mut self) -> Result<TableField> {
        Ok(match self.tokens.peek().map(Spanned::inner) {
            Some(Token::SqrO) => {
                self.tokens.next();
                let index = self.parse_exp()?;
                expect_token!(self.tokens.next(), Token::SqrC);
                expect_token!(self.tokens.next(), Token::Is);
                TableField::Index(index, self.parse_exp()?)
            }
            Some(Token::Ident(_)) => {
                let name = expect_ident!(self.tokens.next());
                expect_token!(self.tokens.next(), Token::Is);
                TableField::Index(Exp::Lit(Value::String(name)), self.parse_exp()?)
            }
            _ => TableField::List(self.parse_exp()?),
        })
    }
}
