use crate::ast::{Ass, Block, Exp, For, Func, If, Local, Parser, PrefixExp, Repeat, While};
use crate::error::{LuaError, Result};
use crate::token::{Pos, Spanned, Token};

/// ```text
/// stat ::=  ‘;’ |
///      varlist ‘=’ explist |
///      functioncall |
///      label |
///      break |
///      goto Name |
///      do block end |
///      while exp do block end |
///      repeat block until exp |
///      if exp then block {elseif exp then block} [else block] end |
///      for Name ‘=’ exp ‘,’ exp [‘,’ exp] do block end |
///      for namelist in explist do block end |
///      function funcname funcbody |
///      local function Name funcbody |
///      local attnamelist [‘=’ explist]
/// label ::= ‘::’ Name ‘::’
/// ```
pub enum Stat {
    Nop,
    Ass(Ass),
    Call(Pos, PrefixExp, Vec<Exp>),
    CallMethod(Pos, Box<PrefixExp>, String, Vec<Exp>),
    Func(String, Func),
    LocalFunc(String, Func),
    Local(Local),
    Do(Block),
    If(If),
    Break,
    While(While),
    Repeat(Repeat),
    For(For),
    Goto(String),
    Label(String),
}

impl<'a> Parser<'a> {
    pub(super) fn parse_stat(&mut self) -> Result<Stat> {
        Ok(match not_eof!(self.tokens.peek()).tuple() {
            (_, Token::Semi) => {
                self.tokens.next();
                Stat::Nop
            }
            (_, Token::Ident(_) | Token::ParO) => {
                let prefix_exp = self.parse_prefix_exp()?;

                match prefix_exp {
                    PrefixExp::Var(..) | PrefixExp::Index(..) => {
                        Stat::Ass(self.parse_ass(prefix_exp)?)
                    }
                    PrefixExp::Call(pos, func, args) => Stat::Call(pos, *func, args),
                    PrefixExp::CallMethod(pos, table, name, args) => {
                        Stat::CallMethod(pos, table, name, args)
                    }
                    _ => panic!(),
                }
            }
            (_, Token::Local) => {
                self.tokens.next();
                match self.tokens.peek().map(Spanned::inner) {
                    Some(Token::Function) => {
                        self.tokens.next();
                        let name = expect_ident!(self.tokens.next());
                        let func = self.parse_func()?;
                        Stat::LocalFunc(name, func)
                    }
                    _ => Stat::Local(self.parse_local()?),
                }
            }
            (_, Token::Function) => {
                self.tokens.next();
                let name = expect_ident!(self.tokens.next());
                let func = self.parse_func()?;
                Stat::Func(name, func)
            }
            (_, Token::Do) => {
                self.tokens.next();
                let block = self.parse_block()?;
                expect_token!(self.tokens.next(), Token::End);
                Stat::Do(block)
            }
            (_, Token::If) => Stat::If(self.parse_if()?),
            (_, Token::Break) => {
                self.tokens.next();
                Stat::Break
            }
            (_, Token::While) => Stat::While(self.parse_while()?),
            (_, Token::Repeat) => Stat::Repeat(self.parse_repeat()?),
            (_, Token::For) => Stat::For(self.parse_for()?),
            (_, Token::Goto) => {
                self.tokens.next();
                let label = expect_ident!(self.tokens.next());
                Stat::Goto(label)
            }
            (_, Token::Ass) => {
                self.tokens.next();
                let label = expect_ident!(self.tokens.next());
                expect_token!(self.tokens.next(), Token::Ass);
                Stat::Label(label)
            }
            (pos, tok) => return err!(pos, LuaError::UnexpectedToken(tok.clone())),
        })
    }
}
