use crate::ast::{Exp, Parser};
use crate::error::{LuaError, Result};
use crate::token::{Pos, Spanned, Token};
use crate::vm::Value;

/// ```text
/// prefixexp ::= var | functioncall | ‘(’ exp ‘)’
/// var ::=  Name | prefixexp ‘[’ exp ‘]’ | prefixexp ‘.’ Name
/// functioncall ::=  prefixexp args | prefixexp ‘:’ Name args
/// args ::=  ‘(’ [explist] ‘)’ | tableconstructor | LiteralString
/// ```
///
/// This means that a prefix expression always starts with either a `Name` or `‘(’`
pub enum PrefixExp {
    Var(String),
    Index(Box<PrefixExp>, Box<Exp>),
    Call(Pos, Box<PrefixExp>, Vec<Exp>),
    CallMethod(Pos, Box<PrefixExp>, String, Vec<Exp>),
    Par(Box<Exp>),
}

impl<'a> Parser<'a> {
    pub(super) fn parse_prefix_exp(&mut self) -> Result<PrefixExp> {
        let (pos, mut res) = match not_eof!(self.tokens.next()).into_tuple() {
            (pos, Token::Ident(name)) => (pos, PrefixExp::Var(name)),
            (pos, Token::ParO) => {
                let res = PrefixExp::Par(Box::new(self.parse_exp()?));
                expect_token!(self.tokens.next(), Token::ParC);
                (pos, res)
            }
            (pos, tok) => return err!(pos, LuaError::UnexpectedToken(tok)),
        };

        loop {
            match self.tokens.peek().map(Spanned::inner) {
                Some(Token::SqrO) => {
                    self.tokens.next();

                    let exp = self.parse_exp()?;

                    expect_token!(self.tokens.next(), Token::SqrC);

                    res = PrefixExp::Index(Box::new(res), Box::new(exp));
                }
                Some(Token::Point) => {
                    self.tokens.next();

                    let ind = expect_ident!(self.tokens.next());
                    res = PrefixExp::Index(Box::new(res), Box::new(Exp::Lit(Value::String(ind))));
                }
                Some(Token::ParO | Token::CurlO | Token::String(..)) => {
                    let args = self.parse_args()?;

                    res = PrefixExp::Call(pos, Box::new(res), args);
                }
                Some(Token::Colon) => {
                    self.tokens.next();

                    let name = expect_ident!(self.tokens.next());
                    let args = self.parse_args()?;

                    res = PrefixExp::CallMethod(pos, Box::new(res), name, args);
                }
                _ => break,
            }
        }

        Ok(res)
    }

    fn parse_args(&mut self) -> Result<Vec<Exp>> {
        match not_eof!(self.tokens.peek()).tuple() {
            (_, Token::ParO) => {
                self.tokens.next();

                let mut args = Vec::new();

                loop {
                    match self.tokens.peek().map(Spanned::inner) {
                        Some(Token::ParC) => break,
                        Some(Token::Comma) => {
                            if args.is_empty() {
                                panic!()
                            } else {
                                self.tokens.next();
                            }
                        }
                        _ => {}
                    }

                    args.push(self.parse_exp()?);
                }

                expect_token!(self.tokens.next(), Token::ParC);

                Ok(args)
            }
            (_, Token::CurlO) => {
                let fields = self.parse_table_constructor()?;
                let exp = Exp::Table(fields);

                Ok(vec![exp])
            }
            (_, Token::String(str)) => {
                let exp = Exp::Lit(Value::String(str.to_string()));
                self.tokens.next();

                Ok(vec![exp])
            }
            (pos, tok) => return err!(pos, LuaError::UnexpectedToken(tok.clone())),
        }
    }
}
