use std::iter::Peekable;

use crate::error::{LuaError, Result};
use crate::token::{Spanned, Tokenizer};

macro_rules! not_eof {
    ( $opt:expr ) => {
        match $opt {
            Some(token) => token,
            None => return err!(crate::error::LuaError::UnexpectedEof),
        }
    };
}

macro_rules! expect_ident {
    ( $spanned:expr ) => {
        match $spanned {
            Some(spanned) => {
                let (pos, tok) = spanned.into_tuple();
                match tok {
                    Token::Ident(s) => s,
                    _ => return err!(pos, crate::error::LuaError::UnexpectedToken(tok)),
                }
            }
            None => return err!(crate::error::LuaError::UnexpectedEof),
        }
    };
}

macro_rules! expect_token {
    ( $spanned:expr, $pat:pat ) => {
        match $spanned {
            Some(spanned) => {
                let (pos, tok) = spanned.into_tuple();
                if !matches!(tok, $pat) {
                    return err!(pos, crate::error::LuaError::UnexpectedToken(tok));
                }
            }
            None => return err!(crate::error::LuaError::UnexpectedEof),
        }
    };
}

mod assignment;
mod block;
mod chunk;
mod expression;
mod r#for;
mod function;
mod r#if;
mod local;
mod prefix_exp;
mod repeat;
mod statement;
mod table;
mod r#while;

pub use assignment::{Ass, Var};
pub use block::Block;
pub use chunk::Chunk;
pub use expression::Exp;
pub use function::Func;
pub use local::{Attr, Local};
pub use prefix_exp::PrefixExp;
pub use r#for::{For, ForGen, ForNum};
pub use r#if::If;
pub use r#while::While;
pub use repeat::Repeat;
pub use statement::Stat;
pub use table::TableField;

/// Parser to parse Lua code into a syntax tree.
/// Entry point for parsing the Lua code is at [`Chunk`].
pub struct Parser<'a> {
    tokens: Peekable<Tokenizer<'a>>,
}

impl<'a> Parser<'a> {
    pub fn parse(input: &'a str) -> Result<Chunk> {
        let mut parser = Self {
            tokens: Tokenizer::read(input).peekable(),
        };

        let chunk = parser.parse_chunk();
        // Ensure all was read
        if let Some((pos, tok)) = parser.tokens.next().map(Spanned::into_tuple) {
            return err!(pos, LuaError::UnexpectedToken(tok));
        }
        chunk
    }
}
