use crate::ast::{Exp, Parser};
use crate::error::Result;
use crate::token::{Spanned, Token};

/// ```text
/// stat ::= local function Name funcbody |
///          local attnamelist [‘=’ explist]
/// attnamelist ::=  Name attrib {‘,’ Name attrib}
/// ```
pub struct Local {
    pub vars: Vec<(String, Option<Attr>)>,
    pub exps: Vec<Exp>,
}

#[derive(Clone, Copy)]
pub enum Attr {
    Const,
    Close,
}

impl<'a> Parser<'a> {
    pub(super) fn parse_local(&mut self) -> Result<Local> {
        let mut vars = vec![self.parse_name_attr()?];
        loop {
            match self.tokens.peek().map(Spanned::inner) {
                Some(Token::Comma) => {
                    self.tokens.next();
                    vars.push(self.parse_name_attr()?);
                }
                Some(Token::Is) => {
                    self.tokens.next();
                    break;
                }
                _ => {
                    return Ok(Local {
                        vars,
                        exps: Vec::new(),
                    })
                }
            }
        }

        let mut exps = vec![self.parse_exp()?];
        while let Some(Token::Comma) = self.tokens.peek().map(Spanned::inner) {
            self.tokens.next();
            exps.push(self.parse_exp()?);
        }

        Ok(Local { vars, exps })
    }

    fn parse_name_attr(&mut self) -> Result<(String, Option<Attr>)> {
        let name = expect_ident!(self.tokens.next());
        let attr = match self.tokens.peek().map(Spanned::inner) {
            Some(Token::Lt) => {
                self.tokens.next();
                let attr = match expect_ident!(self.tokens.next()).as_str() {
                    "const" => Attr::Const,
                    "close" => Attr::Close,
                    _ => panic!(),
                };
                expect_token!(self.tokens.next(), Token::Gt);
                Some(attr)
            }
            _ => None,
        };
        Ok((name, attr))
    }
}
