use crate::ast::{Block, Exp, Parser};
use crate::error::{LuaError, Result};
use crate::token::Token;

/// ```text
/// stat ::= if exp then block {elseif exp then block} [else block] end
/// ```
pub struct If {
    pub conds: Vec<(Exp, Block)>,
    pub els: Option<Box<Block>>,
}

impl<'a> Parser<'a> {
    pub(super) fn parse_if(&mut self) -> Result<If> {
        expect_token!(self.tokens.next(), Token::If);

        let mut conds = vec![self.parse_cond()?];
        let mut els = None;
        loop {
            match not_eof!(self.tokens.next()).into_tuple() {
                (_, Token::Elseif) => conds.push(self.parse_cond()?),
                (_, Token::Else) => {
                    els = Some(Box::new(self.parse_block()?));
                    expect_token!(self.tokens.next(), Token::End);
                    break;
                }
                (_, Token::End) => break,
                (pos, tok) => return err!(pos, LuaError::UnexpectedToken(tok)),
            }
        }

        Ok(If { conds, els })
    }

    fn parse_cond(&mut self) -> Result<(Exp, Block)> {
        let exp = self.parse_exp()?;
        expect_token!(self.tokens.next(), Token::Then);
        let block = self.parse_block()?;
        Ok((exp, block))
    }
}
