use crate::ast::{Block, Parser};
use crate::error::{LuaError, Result};
use crate::token::{Spanned, Token};

/// ```text
/// stat ::= function funcname funcbody
/// functiondef ::= function funcbody
/// funcbody ::= ‘(’ [parlist] ‘)’ block end
/// parlist ::= namelist [‘,’ ‘...’] | ‘...’
/// ```
pub struct Func {
    pub params: Vec<String>,
    pub varargs: bool,
    pub block: Box<Block>,
}

impl<'a> Parser<'a> {
    pub(super) fn parse_func(&mut self) -> Result<Func> {
        let (params, varargs) = self.parse_params()?;

        let block = self.parse_block()?;

        expect_token!(self.tokens.next(), Token::End);

        Ok(Func {
            params,
            varargs,
            block: Box::new(block),
        })
    }

    fn parse_params(&mut self) -> Result<(Vec<String>, bool)> {
        expect_token!(self.tokens.next(), Token::ParO);

        let mut params = Vec::new();
        let mut varargs = false;

        if let Some(Token::ParC) = self.tokens.peek().map(Spanned::inner) {
            self.tokens.next();
            return Ok((params, varargs));
        }

        loop {
            match not_eof!(self.tokens.next()).into_tuple() {
                (_, Token::Ident(name)) => {
                    params.push(name.to_string());
                    match not_eof!(self.tokens.next()).into_tuple() {
                        (_, Token::Comma) => {}
                        (_, Token::ParC) => break,
                        (pos, tok) => return err!(pos, LuaError::UnexpectedToken(tok)),
                    }
                }
                (_, Token::Dots) => {
                    varargs = true;
                    expect_token!(self.tokens.next(), Token::ParC);
                    break;
                }
                (_, Token::ParC) => break,
                (pos, tok) => return err!(pos, LuaError::UnexpectedToken(tok)),
            }
        }

        Ok((params, varargs))
    }
}
