use crate::ast::{Block, Exp, Parser};
use crate::error::{LuaError, Result};
use crate::token::{Pos, Spanned, Token};

/// ```text
/// stat ::= for Name ‘=’ exp ‘,’ exp [‘,’ exp] do block end |
///          for namelist in explist do block end
/// namelist ::= Name {‘,’ Name}
/// explist ::= exp {‘,’ exp}
/// ```

pub enum For {
    Numeric(ForNum),
    Generic(ForGen),
}

pub struct ForNum {
    pub name: String,
    pub init: Box<Exp>,
    pub limit: Box<Exp>,
    pub step: Option<Box<Exp>>,
    pub block: Box<Block>,
}

pub struct ForGen {
    pub pos: Pos,
    pub names: Vec<String>,
    pub exp: Box<Exp>,
    pub block: Box<Block>,
}

impl<'a> Parser<'a> {
    pub(super) fn parse_for(&mut self) -> Result<For> {
        expect_token!(self.tokens.next(), Token::For);

        let name = expect_ident!(self.tokens.next());

        Ok(match not_eof!(self.tokens.peek()).tuple() {
            (_, Token::Is) => For::Numeric(self.parse_for_num(name)?),
            (_, Token::Comma | Token::In) => For::Generic(self.parse_for_gen(name)?),
            (pos, tok) => return err!(pos, LuaError::UnexpectedToken(tok.clone())),
        })
    }

    fn parse_for_num(&mut self, name: String) -> Result<ForNum> {
        expect_token!(self.tokens.next(), Token::Is);

        let init = Box::new(self.parse_exp()?);

        expect_token!(self.tokens.next(), Token::Comma);

        let limit = Box::new(self.parse_exp()?);

        let step = match self.tokens.peek().map(Spanned::inner) {
            Some(Token::Comma) => {
                self.tokens.next();
                Some(Box::new(self.parse_exp()?))
            }
            _ => None,
        };

        expect_token!(self.tokens.next(), Token::Do);

        let block = Box::new(self.parse_block()?);

        expect_token!(self.tokens.next(), Token::End);

        Ok(ForNum {
            name,
            init,
            limit,
            step,
            block,
        })
    }

    fn parse_for_gen(&mut self, name: String) -> Result<ForGen> {
        let mut names = vec![name];

        loop {
            match not_eof!(self.tokens.next()).into_tuple() {
                (_, Token::Comma) => names.push(expect_ident!(self.tokens.next())),
                (_, Token::In) => break,
                (pos, tok) => return err!(pos, LuaError::UnexpectedToken(tok)),
            }
        }

        let pos = self.tokens.peek().map(Spanned::pos).unwrap_or_default();
        let exp = Box::new(self.parse_exp()?);

        expect_token!(self.tokens.next(), Token::Do);

        let block = Box::new(self.parse_block()?);

        expect_token!(self.tokens.next(), Token::End);

        Ok(ForGen {
            pos,
            names,
            exp,
            block,
        })
    }
}
