use crate::ast::{Exp, Parser, PrefixExp};
use crate::error::{LuaError, Result};
use crate::token::{Spanned, Token};

/// ```text
/// stat ::= varlist ‘=’ explist
/// varlist ::= var {‘,’ var}
/// explist ::= exp {‘,’ exp}
/// ```
pub struct Ass {
    pub vars: Vec<Var>,
    pub exps: Vec<Exp>,
}

pub enum Var {
    Name(String),
    Index(PrefixExp, Exp),
}

impl<'a> Parser<'a> {
    pub(super) fn parse_ass(&mut self, first: PrefixExp) -> Result<Ass> {
        let mut vars = vec![self.parse_var(Some(first))?];
        loop {
            match not_eof!(self.tokens.next()).into_tuple() {
                (_, Token::Comma) => vars.push(self.parse_var(None)?),
                (_, Token::Is) => break,
                (pos, tok) => return err!(pos, LuaError::UnexpectedToken(tok)),
            }
        }

        let mut exps = vec![self.parse_exp()?];
        while let Some(Token::Comma) = self.tokens.peek().map(Spanned::inner) {
            self.tokens.next();
            exps.push(self.parse_exp()?);
        }

        Ok(Ass { vars, exps })
    }

    fn parse_var(&mut self, parsed: Option<PrefixExp>) -> Result<Var> {
        let prefix_exp = match parsed {
            Some(exp) => exp,
            None => self.parse_prefix_exp()?,
        };

        Ok(match prefix_exp {
            PrefixExp::Var(name) => Var::Name(name),
            PrefixExp::Index(sub, exp) => Var::Index(*sub, *exp),
            _ => panic!(),
        })
    }
}
