use std::cell::RefCell;
use std::collections::HashMap;
use std::fmt;
use std::hash::{Hash, Hasher};
use std::rc::Rc;

use crate::error::Result;
use crate::vm::VM;

#[derive(Clone, Debug)]
pub enum Value {
    Nil,
    Bool(bool),
    Number(Numeric),
    String(String),
    // TODO: Compare by reference, table index
    Func(Rc<FuncDef>),
    Table(Rc<RefCell<Table>>),
}

#[derive(Clone, Debug)]
pub enum Numeric {
    Integer(i64),
    Float(f64),
}

// TODO: Split into closure and function types
#[derive(Clone, Debug)]
pub struct FuncDef {
    pub params: usize,
    pub varargs: bool,
    pub locals: Vec<VarType>,
    pub r#impl: FuncImpl,
}

#[derive(Clone, Debug)]
pub enum VarType {
    Local,
    Param,
    Up(usize),
}

#[derive(Clone)]
pub enum FuncImpl {
    Builtin {
        name: &'static str,
        func: fn(&mut VM) -> Result<VmValue>,
    },
    Defined(usize),
}

#[derive(Clone, Debug, Default)]
pub struct Table {
    map: HashMap<Value, Value>,
    next_key: Option<HashMap<Value, Value>>,
}

/// Used to implement returning multiple values
#[derive(Clone, Debug)]
pub enum VmValue {
    Single(Value),
    Multiple(Vec<Value>),
}

impl Value {
    pub fn int(n: i64) -> Self {
        Self::Number(Numeric::Integer(n))
    }

    pub fn float(f: f64) -> Self {
        Self::Number(Numeric::Float(f))
    }

    pub fn is_falsy(&self) -> bool {
        matches!(self, Value::Nil | Value::Bool(false))
    }

    pub fn is_truthy(&self) -> bool {
        !self.is_falsy()
    }
}

impl fmt::Display for Value {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Value::Nil => write!(f, "nil"),
            Value::Bool(bool) => write!(f, "{}", bool),
            Value::Number(num) => write!(f, "{}", num),
            Value::String(str) => write!(f, "{}", str),
            Value::Func(func) => write!(f, "function: {:?}", Rc::as_ptr(func)),
            Value::Table(tbl) => write!(f, "table: {:?}", Rc::as_ptr(tbl)),
        }
    }
}

impl Hash for Value {
    fn hash<H: Hasher>(&self, state: &mut H) {
        match self {
            Value::Nil => state.write_u8(0),
            Value::Bool(bool) => {
                state.write_u8(1);
                bool.hash(state);
            }
            Value::Number(num) => {
                state.write_u8(2);
                num.hash(state);
            }
            Value::String(str) => {
                state.write_u8(3);
                str.hash(state);
            }
            Value::Func(func) => {
                state.write_u8(4);
                Rc::as_ptr(func).hash(state);
            }
            Value::Table(tbl) => {
                state.write_u8(5);
                Rc::as_ptr(tbl).hash(state);
            }
        }
    }
}

impl PartialEq for Value {
    fn eq(&self, other: &Self) -> bool {
        match (&self, &other) {
            (&Value::Nil, &Value::Nil) => true,
            (&Value::Bool(b1), &Value::Bool(b2)) => b1 == b2,
            (&Value::Number(n1), &Value::Number(n2)) => n1 == n2,
            (&Value::String(s1), &Value::String(s2)) => s1 == s2,
            (&Value::Func(f1), &Value::Func(f2)) => Rc::as_ptr(f1) == Rc::as_ptr(f2),
            (&Value::Table(v1), &Value::Table(v2)) => Rc::as_ptr(v1) == Rc::as_ptr(v2),
            _ => false,
        }
    }
}

impl Eq for Value {}

impl Numeric {
    pub(super) fn is_int(&self) -> bool {
        matches!(self, Numeric::Integer(..))
    }

    pub(super) fn to_int(&self) -> i64 {
        match self {
            Numeric::Integer(n) => *n,
            Numeric::Float(n) => {
                if n - n.floor() <= 10e-6 {
                    n.floor() as i64
                } else {
                    panic!()
                }
            }
        }
    }

    pub(super) fn to_float(&self) -> f64 {
        match self {
            Numeric::Integer(n) => *n as f64,
            Numeric::Float(n) => *n,
        }
    }
}

impl fmt::Display for Numeric {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Numeric::Integer(n) => write!(f, "{}", n),
            Numeric::Float(n) => write!(f, "{}", n),
        }
    }
}

impl Hash for Numeric {
    fn hash<H: Hasher>(&self, state: &mut H) {
        match self {
            Numeric::Integer(n) => {
                state.write_u8(0);
                state.write_i64(*n);
            }
            Numeric::Float(f) => {
                state.write_u8(1);
                for byte in f.to_be_bytes() {
                    state.write_u8(byte);
                }
            }
        }
    }
}

impl PartialEq for Numeric {
    fn eq(&self, other: &Self) -> bool {
        match (self, other) {
            (&Numeric::Integer(n1), &Numeric::Integer(n2)) => n1 == n2,
            (&Numeric::Float(f1), &Numeric::Float(f2)) => f1 == f2,
            _ => false,
        }
    }
}

impl Eq for Numeric {}

impl fmt::Debug for FuncImpl {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            FuncImpl::Builtin { name, .. } => {
                write!(f, "builtin:{}", name)
            }
            FuncImpl::Defined(n) => write!(f, "defined:{}", n),
        }
    }
}

impl Table {
    pub(super) fn init_next_keys(&mut self) -> Value {
        let keys: Vec<Value> = self.map.keys().cloned().collect();
        self.next_key = Some(
            keys.iter()
                .cloned()
                .zip(keys.iter().skip(1).cloned())
                .collect(),
        );
        keys.first().cloned().unwrap_or(Value::Nil)
    }

    pub(super) fn next_key(&mut self, key: Value) -> Value {
        self.next_key
            .as_ref()
            .unwrap()
            .get(&key)
            .cloned()
            .unwrap_or(Value::Nil)
    }

    pub(super) fn border(&self) -> usize {
        let mut border = 1;
        while let Some(..) = self.map.get(&Value::int(border as i64)) {
            border += 1;
        }
        border - 1
    }

    pub(super) fn get(&self, key: Value) -> Value {
        self.map.get(&key).cloned().unwrap_or(Value::Nil)
    }

    pub(super) fn set(&mut self, key: Value, value: Value) {
        // TODO: Convert int-valued-float key to int
        if let Value::Nil = value {
            self.map.remove(&key);
        } else {
            self.map.insert(key, value);
        }
    }
}

impl VmValue {
    pub fn empty() -> Self {
        VmValue::Multiple(vec![])
    }

    pub fn single(self) -> Value {
        match self {
            VmValue::Single(val) => val.clone(),
            VmValue::Multiple(vec) => vec.get(0).cloned().unwrap_or(Value::Nil),
        }
    }
}
