use std::ops::{Add, BitAnd, BitOr, BitXor, Div, Mul, Rem, Shl, Shr, Sub};

use crate::vm::{Numeric, Value};

#[derive(Clone, Copy, Debug)]
pub enum BinOp {
    // Mathematical
    Add,
    Sub,
    Mul,
    Div,
    FlDiv,
    Pow,
    Mod,
    // Bitwise
    BitAnd,
    BitXor,
    BitOr,
    Shl,
    Shr,
    // String
    Conc,
    // Comparison
    Eq,
    Neq,
    Lt,
    Leq,
    Gt,
    Geq,
    // Logical
    And,
    Or,
}

#[derive(Clone, Copy, Debug)]
pub enum UnOp {
    Minus,
    BitNot,
    Not,
    Len,
}

impl BinOp {
    pub fn precedence(&self) -> usize {
        // Value '10' is absent since that is given to unary operators
        match self {
            BinOp::Or => 0,
            BinOp::And => 1,
            BinOp::Lt | BinOp::Leq | BinOp::Gt | BinOp::Geq | BinOp::Eq | BinOp::Neq => 2,
            BinOp::BitOr => 3,
            BinOp::BitXor => 4,
            BinOp::BitAnd => 5,
            BinOp::Shl | BinOp::Shr => 6,
            BinOp::Conc => 7,
            BinOp::Add | BinOp::Sub => 8,
            BinOp::Mul | BinOp::Div | BinOp::FlDiv | BinOp::Mod => 9,
            BinOp::Pow => 11,
        }
    }

    pub fn is_left_ass(&self) -> bool {
        !matches!(self, BinOp::Conc | BinOp::Pow)
    }
}

impl UnOp {
    pub fn precedence(&self) -> usize {
        10 // Precedence is always 10, above all binary operators except 'pow'
    }
}

// TODO: Check spec for exact implementation, especially regarding wrapping
macro_rules! math_op {
    ( $op:ident, $op1:expr, $op2:expr ) => {{
        if let (Value::Number(n1), Value::Number(n2)) = ($op1, $op2) {
            Value::Number(n1.$op(n2))
        } else {
            panic!("both operands must be numbers")
        }
    }};
}

macro_rules! comp_op {
    ( $op:ident, $op1:expr, $op2:expr ) => {{
        match ($op1, $op2) {
            (Value::Number(n1), Value::Number(n2)) => {
                Value::Bool(n1.to_float().$op(&n2.to_float()))
            }
            (Value::String(s1), Value::String(s2)) => Value::Bool(s1.$op(&s2)),
            _ => panic!(),
        }
    }};
}

impl Value {
    // TODO: Make this work with references?
    pub(super) fn bin_op(self, op: BinOp, rhs: Value) -> Value {
        match op {
            // Mathematical
            BinOp::Add => math_op!(add, self, rhs),
            BinOp::Sub => math_op!(sub, self, rhs),
            BinOp::Mul => math_op!(mul, self, rhs),
            BinOp::Div => math_op!(div, self, rhs),
            BinOp::FlDiv => math_op!(fldiv, self, rhs),
            BinOp::Mod => math_op!(rem, self, rhs),
            BinOp::Pow => math_op!(pow, self, rhs),
            // Bitwise
            BinOp::BitAnd => math_op!(bitand, self, rhs),
            BinOp::BitOr => math_op!(bitor, self, rhs),
            BinOp::BitXor => math_op!(bitxor, self, rhs),
            BinOp::Shl => math_op!(shl, self, rhs),
            BinOp::Shr => math_op!(shr, self, rhs),
            // String
            BinOp::Conc => {
                let op1 = match self {
                    Value::Number(n) => format!("{}", n),
                    Value::String(str) => str.to_string(),
                    _ => panic!(),
                };
                let op2 = match rhs {
                    Value::Number(n) => format!("{}", n),
                    Value::String(str) => str.to_string(),
                    _ => panic!(),
                };
                Value::String(format!("{}{}", op1, op2))
            }
            // Comparison
            BinOp::Eq => Value::Bool(match (self, rhs) {
                (Value::Nil, Value::Nil) => true,
                (Value::Bool(b1), Value::Bool(b2)) => b1 == b2,
                (Value::Number(n1), Value::Number(n2)) => {
                    (n1.to_float() - n2.to_float()).abs() < 10e-8
                }
                (Value::String(s1), Value::String(s2)) => s1 == s2,
                _ => false,
            }),
            BinOp::Neq => {
                if let Value::Bool(bool) = self.bin_op(BinOp::Eq, rhs) {
                    Value::Bool(!bool)
                } else {
                    panic!()
                }
            }
            BinOp::Lt => comp_op!(lt, self, rhs),
            BinOp::Leq => comp_op!(le, self, rhs),
            BinOp::Gt => comp_op!(gt, self, rhs),
            BinOp::Geq => comp_op!(ge, self, rhs),
            // Logical, short circuiting implemented in compiler
            BinOp::And | BinOp::Or => unreachable!(),
        }
    }

    pub(super) fn un_op(self, op: UnOp) -> Value {
        match op {
            UnOp::Minus => match self {
                Value::Number(n) => Value::Number(n.minus()),
                _ => panic!("operand must be number"),
            },
            UnOp::BitNot => match self {
                Value::Number(n) => Value::Number(n.bitnot()),
                _ => panic!("operand must be number"),
            },
            UnOp::Not => Value::Bool(self.is_falsy()),
            UnOp::Len => match self {
                Value::String(s) => Value::int(s.len() as i64),
                Value::Table(t) => Value::int(t.borrow().border() as i64),
                _ => panic!("operand must be string or table"),
            },
        }
    }
}

macro_rules! numeric_op {
    ( $op:ident ) => {
        fn $op(&self, rhs: Self) -> Self {
            if self.is_int() && rhs.is_int() {
                Numeric::Integer(self.to_int().$op(rhs.to_int()))
            } else {
                Numeric::Float(self.to_float().$op(rhs.to_float()))
            }
        }
    };
}

macro_rules! numeric_bit_op {
    ( $op:ident ) => {
        fn $op(&self, rhs: Self) -> Self {
            Numeric::Integer(self.to_int().$op(rhs.to_int()))
        }
    };
}

impl Numeric {
    numeric_op!(add);
    numeric_op!(sub);
    numeric_op!(mul);
    numeric_op!(rem);

    fn div(&self, rhs: Self) -> Self {
        Numeric::Float(self.to_float().div(rhs.to_float()))
    }

    fn fldiv(&self, rhs: Self) -> Self {
        if self.is_int() && rhs.is_int() {
            Numeric::Integer(self.to_int().div(rhs.to_int()))
        } else {
            Numeric::Float(self.to_float().div(rhs.to_float()).floor())
        }
    }

    fn pow(&self, rhs: Self) -> Self {
        Numeric::Float(self.to_float().powf(rhs.to_float()))
    }

    numeric_bit_op!(bitand);
    numeric_bit_op!(bitor);
    numeric_bit_op!(bitxor);
    numeric_bit_op!(shl);
    numeric_bit_op!(shr);

    fn bitnot(&self) -> Self {
        Numeric::Integer(!self.to_int())
    }

    fn minus(&self) -> Self {
        match self {
            Numeric::Integer(n) => Numeric::Integer(-n),
            Numeric::Float(f) => Numeric::Float(-f),
        }
    }
}
