use std::cmp::Ordering;
use std::convert::TryInto;

use rand::{Rng, SeedableRng};
use rand_xoshiro::Xoshiro256StarStar;

use crate::error::Result;
use crate::vm::{BinOp, Numeric, Value, VmValue, VM};

macro_rules! num {
    ( $x:expr ) => {
        match $x {
            Value::Number(n) => n,
            _ => panic!("not a number"),
        }
    };
}

macro_rules! int {
    ( $x:expr ) => {
        match $x {
            Value::Number(n) => n.to_int(),
            _ => panic!("not a number"),
        }
    };
}

macro_rules! float {
    ( $x:expr ) => {
        match $x {
            Value::Number(n) => n.to_float(),
            _ => panic!("not a number"),
        }
    };
}

pub(super) fn abs(x: Value) -> Result<VmValue> {
    match x {
        Value::Number(n) => Ok(VmValue::Single(Value::Number(match n {
            Numeric::Integer(i) => Numeric::Integer(i.abs()),
            Numeric::Float(f) => Numeric::Float(f.abs()),
        }))),
        _ => panic!("not a number"),
    }
}

pub(super) fn acos(x: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::float(float!(x).acos())))
}

pub(super) fn asin(x: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::float(float!(x).asin())))
}

pub(super) fn atan(y: Value, x: Value) -> Result<VmValue> {
    let y = float!(y);
    Ok(VmValue::Single(Value::float(match x {
        Value::Number(n) => y.atan2(n.to_float()),
        Value::Nil => y.atan(),
        _ => panic!("not a number"),
    })))
}

pub(super) fn ceil(x: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::int(float!(x).ceil() as i64)))
}

pub(super) fn cos(x: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::float(float!(x).cos())))
}

pub(super) fn deg(x: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::float(float!(x).to_degrees())))
}

pub(super) fn exp(x: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::float(float!(x).exp())))
}

pub(super) fn floor(x: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::int(float!(x).floor() as i64)))
}

pub(super) fn fmod(x: Value, y: Value) -> Result<VmValue> {
    let x = num!(x);
    let y = num!(y);
    Ok(VmValue::Single(match (&x, &y) {
        (Numeric::Integer(x), Numeric::Integer(y)) => Value::int(x % y),
        _ => Value::float(x.to_float() % y.to_float()),
    }))
}

pub(super) const HUGE: f64 = std::f64::INFINITY;

pub(super) fn log(x: Value, base: Value) -> Result<VmValue> {
    let x = float!(x);
    Ok(VmValue::Single(Value::float(match base {
        Value::Number(n) => x.log(n.to_float()),
        Value::Nil => x.ln(),
        _ => panic!("not a number"),
    })))
}

pub(super) fn max(x: Value, rest: Vec<Value>) -> Result<VmValue> {
    let mut max = x;
    for val in rest.into_iter() {
        if max.clone().bin_op(BinOp::Lt, val.clone()).is_truthy() {
            max = val;
        }
    }
    Ok(VmValue::Single(max))
}

pub(super) const MAX_INTEGER: i64 = std::i64::MAX;

pub(super) fn min(x: Value, rest: Vec<Value>) -> Result<VmValue> {
    let mut min = x;
    for val in rest.into_iter() {
        if val.clone().bin_op(BinOp::Lt, min.clone()).is_truthy() {
            min = val;
        }
    }
    Ok(VmValue::Single(min))
}

pub(super) const MIN_INTEGER: i64 = std::i64::MIN;

pub(super) fn modf(x: Value) -> Result<VmValue> {
    let x = match x {
        Value::Number(n) => n.to_float(),
        _ => panic!("not a number"),
    };
    Ok(VmValue::Multiple(vec![
        Value::int(x.trunc() as i64),
        Value::float(x.fract()),
    ]))
}

pub(super) const PI: f64 = std::f64::consts::PI;

pub(super) fn rad(x: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::float(float!(x).to_radians())))
}

pub(super) fn random(vm: &mut VM, m: Value, n: Value) -> Result<VmValue> {
    Ok(VmValue::Single(match m {
        Value::Number(lower) => {
            let lower = lower.to_int();
            match n {
                Value::Number(upper) => Value::int(vm.rng.gen_range(lower..upper.to_int() + 1)),
                Value::Nil => match lower.cmp(&0) {
                    Ordering::Greater => Value::int(vm.rng.gen_range(0..lower) + 1),
                    Ordering::Equal => Value::int(vm.rng.gen()),
                    Ordering::Less => panic!(),
                },
                _ => panic!("not a number"),
            }
        }
        Value::Nil => Value::float(vm.rng.gen()),
        _ => panic!("not a number"),
    }))
}

pub(super) fn randomseed(vm: &mut VM, x: Value, y: Value) -> Result<VmValue> {
    let x = match x {
        Value::Number(n) => n.to_int(),
        Value::Nil => match random(vm, Value::int(0), Value::Nil).unwrap().single() {
            Value::Number(n) => n.to_int(),
            _ => panic!("not a number"),
        },
        _ => panic!("not a number"),
    };
    let y = match y {
        Value::Number(n) => n.to_int(),
        Value::Nil => 0,
        _ => panic!("not a number"),
    };
    let seed: Vec<u8> = [
        x.to_ne_bytes(),
        0xffu64.to_ne_bytes(),
        y.to_ne_bytes(),
        0u64.to_ne_bytes(),
    ]
    .concat();
    vm.rng = Xoshiro256StarStar::from_seed(seed.try_into().unwrap());
    // Discard some values (reference implementation does this)
    for _ in 0..16 {
        random(vm, Value::int(0), Value::Nil).unwrap();
    }
    Ok(VmValue::Multiple(vec![Value::int(x), Value::int(y)]))
}

pub(super) fn sin(x: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::float(float!(x).sin())))
}

pub(super) fn sqrt(x: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::float(float!(x).sqrt())))
}

pub(super) fn tan(x: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::float(float!(x).tan())))
}

pub(super) fn tointeger(x: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::int(int!(x))))
}

pub(super) fn r#type(x: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::String(
        match num!(x) {
            Numeric::Integer(..) => "integer",
            Numeric::Float(..) => "float",
        }
        .to_string(),
    )))
}

pub(super) fn ult(m: Value, n: Value) -> Result<VmValue> {
    Ok(VmValue::Single(Value::Bool(
        (int!(m) as u64) < int!(n) as u64,
    )))
}
