use crate::error::Result;
use crate::vm::{Numeric, Value, VmValue, VM};

pub(super) fn print(vm: &mut VM, args: Vec<Value>) -> Result<VmValue> {
    let mut iter = args.iter();
    if let Some(first) = iter.next() {
        write!(vm.out, "{}", first)?;
    }
    for val in iter {
        write!(vm.out, " {}", val)?;
    }

    writeln!(vm.out)?;
    Ok(VmValue::empty())
}

pub(super) fn error(_: Vec<Value>) -> Result<VmValue> {
    panic!("invoked error()")
}

pub(super) fn ipairs(vm: &mut VM, t: Value) -> Result<VmValue> {
    let iter = vm
        .env
        .borrow()
        .get(Value::String("__ipairs_next".to_string()))
        .clone();
    Ok(VmValue::Multiple(vec![iter, t, Value::int(0)]))
}

pub(super) fn __ipairs_next(table: Value, index: Value) -> Result<VmValue> {
    let table = match table {
        Value::Table(t) => t,
        _ => panic!("not a table"),
    };
    let table = table.borrow();
    let next = match index {
        Value::Number(Numeric::Integer(n)) => Value::int(n + 1),
        _ => panic!("invalid state"),
    };
    Ok(VmValue::Multiple(match table.get(next.clone()) {
        Value::Nil => vec![Value::Nil],
        val => vec![next, val.clone()],
    }))
}

pub(super) fn next(table: Value, index: Value) -> Result<VmValue> {
    let table = match table {
        Value::Table(t) => t,
        _ => panic!("not a table"),
    };
    let mut table = table.borrow_mut();
    let next = match index {
        Value::Nil => table.init_next_keys(),
        v => table.next_key(v),
    };
    let val = table.get(next.clone());
    Ok(VmValue::Multiple(vec![next, val]))
}

pub(super) fn pairs(vm: &mut VM, t: Value) -> Result<VmValue> {
    let iter = vm
        .env
        .borrow()
        .get(Value::String("next".to_string()))
        .clone();
    Ok(VmValue::Multiple(vec![iter, t, Value::Nil]))
}
