use crate::ast::Stat;
use crate::compiler::{Compiler, ScopeType};
use crate::vm::{OpCode, Value};

impl Compiler {
    pub(super) fn compile_stat(&mut self, stat: Stat) {
        match stat {
            Stat::Nop => {}
            Stat::Ass(ass) => self.compile_ass(ass),
            Stat::Call(exp, args) => {
                self.compile_call(exp, args);
                self.emit(OpCode::PopDisc);
            }
            Stat::CallMethod(prefix, name, args) => {
                self.compile_call_method(*prefix, name, args);
                self.emit(OpCode::PopDisc);
            }
            Stat::Func(name, func) => {
                self.compile_func(func);
                match self.scopes.get(&name) {
                    Some((dst, _)) => self.emit(OpCode::PopLocal(dst)),
                    None => {
                        self.emit(OpCode::PushLit(Value::String(name)));
                        self.emit(OpCode::PushEnv);
                        self.emit(OpCode::SetTbl);
                    }
                }
            }
            Stat::LocalFunc(name, func) => {
                self.compile_func(func);
                let dst = self.scopes.declare_local(name, None);
                self.emit(OpCode::PopLocal(dst));
            }
            Stat::Local(local) => self.compile_local(local),
            Stat::Do(block) => {
                self.scope_enter(ScopeType::Light);

                self.compile_block(block);

                self.scope_leave();
            }
            Stat::If(iff) => {
                self.compile_if(iff);
            }
            Stat::Break => {
                // TODO: Maybe jump to the jump-to-end of the loop since its position is already known?

                // Unroll the stack including scope of the loop
                let depth = match self.scopes.break_depth() {
                    Some(d) => d,
                    None => panic!("break outside of loop"),
                };
                // Unroll once more than depth
                for _ in 0..depth + 1 {
                    self.emit(OpCode::ScopeLeave);
                }

                // Jump to end of loop, placeholder
                let pos = self.pos();
                self.emit(OpCode::Jump(0));
                self.scopes.break_save(pos);
            }
            Stat::While(r#while) => {
                self.compile_while(r#while);
            }
            Stat::Repeat(repeat) => {
                self.compile_repeat(repeat);
            }
            Stat::For(r#for) => {
                self.compile_for(r#for);
            }
            Stat::Goto(label) => {
                self.scopes.goto_insert(label, 0, self.pos());
                self.emit(OpCode::Goto(0, 0)); // Placeholder
            }
            Stat::Label(label) => {
                self.scopes.label_insert(label, self.pos());
            }
        }
    }
}
