use std::collections::HashMap;

mod assignment;
mod block;
mod chunk;
mod expression;
mod r#for;
mod function;
mod r#if;
mod local;
mod prefix_exp;
mod repeat;
mod statement;
mod r#while;

use crate::ast::{Attr, Chunk};
use crate::vm::{OpCode, VarType};

pub struct Compiler {
    // Tracking local variables
    scopes: Scopes,
    chunks: Vec<Vec<OpCode>>,
    current: usize,
}

impl Compiler {
    pub fn compile(chunk: Chunk) -> Vec<Vec<OpCode>> {
        let mut compiler = Self {
            scopes: Scopes::new(),
            chunks: vec![Vec::new()],
            current: 0,
        };

        compiler.compile_chunk(chunk);

        compiler.chunks
    }

    fn emit(&mut self, op: OpCode) {
        self.chunks[self.current].push(op);
    }

    fn emit_all<I: IntoIterator<Item = OpCode>>(&mut self, code: I) {
        self.chunks[self.current].extend(code);
    }

    fn pos(&self) -> (usize, usize) {
        (self.current, self.chunks[self.current].len())
    }

    fn set(&mut self, pos: (usize, usize), op: OpCode) {
        self.chunks[pos.0][pos.1] = op;
    }

    fn scope_enter(&mut self, typ: ScopeType) {
        let enter = match typ {
            ScopeType::Func => None,
            _ => {
                let pos = self.pos();
                self.emit(OpCode::Jump(0)); // Placeholder
                Some(pos)
            }
        };

        self.scopes.open(typ, enter);
    }

    fn scope_leave(&mut self) -> Scope {
        let mut scope = self.scopes.close();

        if let Some(enter) = scope.enter.take() {
            self.emit(OpCode::ScopeLeave);
            self.set(
                enter,
                OpCode::ScopeEnter(
                    scope
                        .locals
                        .split_off(0)
                        .into_iter()
                        .map(|(t, _)| t)
                        .collect(),
                ),
            );
        }

        for break_pos in scope.breaks.split_off(0).into_iter() {
            let off = offset(break_pos, self.pos());
            self.set(break_pos, OpCode::Jump(off));
        }

        for (label, depth, goto_pos) in scope.gotos.split_off(0).into_iter() {
            match scope.labels.get(&label) {
                Some(label_pos) => {
                    let off = offset(goto_pos, *label_pos);
                    self.set(goto_pos, OpCode::Goto(depth, off));
                }
                // Bubble to parent scope, except if current scope is function
                None => {
                    if matches!(scope.typ, ScopeType::Func) {
                        panic!("no label found in scope for goto")
                    } else {
                        self.scopes.goto_insert(label, depth + 1, goto_pos);
                    }
                }
            };
        }

        scope
    }
}

struct Scopes {
    scopes: Vec<Scope>,
}

impl Scopes {
    fn new() -> Self {
        Self { scopes: Vec::new() }
    }

    fn open(&mut self, typ: ScopeType, enter: Option<(usize, usize)>) {
        self.scopes.push(Scope::new(typ, enter));
    }

    fn close(&mut self) -> Scope {
        self.scopes.pop().unwrap()
    }

    fn get(&mut self, name: &str) -> Option<(usize, Option<Attr>)> {
        // Find variable
        for (depth, scope) in self.scopes.iter().rev().enumerate() {
            if let Some(local) = scope.dict.get(name) {
                let attr = scope.locals[*local].1;
                if depth == 0 {
                    // Local found in current scope, just return
                    return Some((*local, attr));
                } else {
                    // Insert upvalues
                    let len = self.scopes.len();
                    let mut local = *local;
                    for scope in self.scopes.iter_mut().skip(len - depth) {
                        let up = VarType::Up(local);
                        local = scope.declare(name.to_string(), up, attr);
                    }
                    return Some((local, attr));
                }
            }
        }
        None
    }

    fn declare_local(&mut self, name: String, attr: Option<Attr>) -> usize {
        self.scopes
            .last_mut()
            .unwrap()
            .declare(name, VarType::Local, attr)
    }

    fn declare_params(&mut self, params: Vec<String>) {
        let scope = self.scopes.last_mut().unwrap();
        for param in params.into_iter() {
            scope.declare(param, VarType::Param, None);
        }
    }

    fn break_depth(&self) -> Option<usize> {
        for (i, scope) in self.scopes.iter().rev().enumerate() {
            match scope.typ {
                ScopeType::Loop => return Some(i),
                ScopeType::Func => return None,
                ScopeType::Light => {}
            }
        }
        None
    }

    fn break_save(&mut self, pos: (usize, usize)) {
        for scope in self.scopes.iter_mut().rev() {
            match scope.typ {
                ScopeType::Loop => {
                    scope.breaks.push(pos);
                }
                ScopeType::Func => return,
                ScopeType::Light => {}
            }
        }
    }

    fn goto_insert(&mut self, name: String, depth: usize, pos: (usize, usize)) {
        let scope = self.scopes.last_mut().unwrap();
        scope.gotos.push((name, depth, pos));
    }

    fn label_insert(&mut self, name: String, pos: (usize, usize)) {
        let scope = self.scopes.last_mut().unwrap();
        scope.labels.insert(name, pos);
    }
}

enum ScopeType {
    Light,
    Func,
    Loop,
}

pub struct Scope {
    // Management
    typ: ScopeType,
    enter: Option<(usize, usize)>,
    // Locals
    locals: Vec<(VarType, Option<Attr>)>,
    dict: HashMap<String, usize>,
    // Breaks/gotos
    breaks: Vec<(usize, usize)>,
    gotos: Vec<(String, usize, (usize, usize))>,
    labels: HashMap<String, (usize, usize)>,
}

impl Scope {
    fn new(typ: ScopeType, enter: Option<(usize, usize)>) -> Self {
        Self {
            typ,
            enter,
            locals: Vec::new(),
            dict: HashMap::new(),
            breaks: Vec::new(),
            gotos: Vec::new(),
            labels: HashMap::new(),
        }
    }

    fn declare(&mut self, name: String, typ: VarType, attr: Option<Attr>) -> usize {
        let ind = self.locals.len();
        self.locals.push((typ, attr));
        self.dict.insert(name, ind);
        ind
    }
}

/// Expects position of jump and position to jump to
fn offset(jump: (usize, usize), target: (usize, usize)) -> i64 {
    assert_eq!(jump.0, target.0);
    // Compiler::pos() gets the position of the next emitted opcode,
    // usually one would save the result from pos(),
    // emit a placeholder, and later use the saved pos to fill placeholder position
    // This means that we need to adjust by one here
    target.1 as i64 - jump.1 as i64 - 1
}
