use crate::ast::If;
use crate::compiler::{offset, Compiler, ScopeType};
use crate::vm::OpCode;

impl Compiler {
    pub(super) fn compile_if(&mut self, iff: If) {
        let mut jump_to_ends = Vec::new();

        // if and elseif blocks
        for (exp, block) in iff.conds.into_iter() {
            self.compile_exp(exp);

            // Jump to next block if condition is false, placeholder
            let jump_next = self.pos();
            self.emit(OpCode::Jump(0)); // Placeholder

            self.scope_enter(ScopeType::Light);

            self.compile_block(block);

            self.scope_leave();

            // Jump to end, placeholder
            // In case of a single if block this is redundant, but don't care
            let jump_end = self.pos();
            self.emit(OpCode::Jump(0));
            jump_to_ends.push(jump_end);

            // Fill jump to next block
            let offset = offset(jump_next, self.pos());
            self.set(jump_next, OpCode::JumpIfNot(offset));
        }

        // else block
        if let Some(block) = iff.els {
            self.scope_enter(ScopeType::Light);

            self.compile_block(*block);

            self.scope_leave();
        }

        // resolve jump to ends
        let end = self.pos();
        for jump in jump_to_ends.into_iter() {
            let offset = offset(jump, end);
            self.set(jump, OpCode::Jump(offset));
        }
    }
}
