use std::rc::Rc;

use crate::ast::{Exp, Func, PrefixExp};
use crate::compiler::{Compiler, ScopeType};
use crate::vm::{FuncDef, FuncImpl, OpCode, Value};

impl Compiler {
    pub(super) fn compile_func(&mut self, func: Func) {
        self.scope_enter(ScopeType::Func);
        let param_count = func.params.len();
        self.scopes.declare_params(func.params);

        // Compile into new chunk
        let label = self.chunks.len();
        let current_save = self.current;
        self.current = label;
        self.chunks.push(Vec::new());
        self.compile_block(*func.block);
        self.emit(OpCode::Combine(0)); // Build empty return value
        self.emit(OpCode::Return); // Ensure that function has at least one return
        self.current = current_save;

        let scope = self.scope_leave();
        let def = FuncDef {
            params: param_count,
            varargs: func.varargs,
            locals: scope.locals.into_iter().map(|(t, _)| t).collect(),
            r#impl: FuncImpl::Defined(label),
        };

        self.emit(OpCode::PushLit(Value::Func(Rc::new(def))));
        self.emit(OpCode::Closure);
    }

    pub(super) fn compile_call(&mut self, prefix: PrefixExp, args: Vec<Exp>) {
        let len = args.len();
        for arg in args.into_iter().rev() {
            self.compile_exp(arg);
        }

        self.compile_prefix_exp(prefix);

        self.emit(OpCode::Call(len));
    }

    pub(super) fn compile_call_method(&mut self, prefix: PrefixExp, name: String, args: Vec<Exp>) {
        let len = args.len();
        for arg in args.into_iter().rev() {
            self.compile_exp(arg);
        }

        self.compile_prefix_exp(prefix);
        self.emit(OpCode::Copy);
        self.emit(OpCode::PushLit(Value::String(name)));
        self.emit(OpCode::GetTbl);

        self.emit(OpCode::Call(len + 1));
    }
}
