use crate::ast::{For, ForGen, ForNum};
use crate::compiler::{offset, Compiler, ScopeType};
use crate::vm::{BinOp, OpCode, Value};

impl Compiler {
    pub(super) fn compile_for(&mut self, r#for: For) {
        match r#for {
            For::Numeric(f) => self.compile_for_num(f),
            For::Generic(f) => self.compile_for_gen(f),
        }
    }

    fn compile_for_num(&mut self, r#for: ForNum) {
        self.scope_enter(ScopeType::Loop);

        // Declare control variable and set to init
        let ctrl = self.scopes.declare_local(r#for.name, None);
        self.compile_exp(*r#for.init);
        self.emit(OpCode::PopLocal(ctrl));

        // Declare (hidden) limit variable and set
        let limit = self.scopes.declare_local("--limit".to_string(), None);
        self.compile_exp(*r#for.limit);
        self.emit(OpCode::PopLocal(limit));

        // Declare (hidden) step variable and set
        let step = self.scopes.declare_local("--step".to_string(), None);
        match r#for.step {
            Some(step) => self.compile_exp(*step),
            None => self.emit(OpCode::PushLit(Value::int(1))),
        }
        self.emit(OpCode::PopLocal(step));

        // Check limit
        let start = self.pos();
        self.emit(OpCode::PushLocal(ctrl));
        self.emit(OpCode::PushLocal(limit));
        self.emit(OpCode::BinOp(BinOp::Leq));

        // Jump to end if false, placeholder
        let jump_end = self.pos();
        self.emit(OpCode::Jump(0)); // Placeholder

        // Block
        self.compile_block(*r#for.block);

        // Step
        self.emit(OpCode::PushLocal(ctrl));
        self.emit(OpCode::PushLocal(step));
        self.emit(OpCode::BinOp(BinOp::Add));
        self.emit(OpCode::PopLocal(ctrl));
        let off = offset(self.pos(), start);
        self.emit(OpCode::Jump(off));

        // Fill in jump to end
        let off = offset(jump_end, self.pos());
        self.set(jump_end, OpCode::JumpIfNot(off));

        self.scope_leave();
    }

    fn compile_for_gen(&mut self, r#for: ForGen) {
        self.scope_enter(ScopeType::Loop);

        // Declare variables
        let mut vars = Vec::new();
        for name in r#for.names.into_iter() {
            vars.push(self.scopes.declare_local(name, None));
        }

        // Evaluate expressions
        self.compile_exp(*r#for.exp);
        let ass = self.scopes.declare_local("--ass".to_string(), None);
        self.emit(OpCode::PopLocal(ass));

        // Initialize variables
        let iter = self.scopes.declare_local("--iter".to_string(), None);
        self.emit(OpCode::PushLocalMult(ass, 0));
        self.emit(OpCode::PopLocal(iter));

        let state = self.scopes.declare_local("--state".to_string(), None);
        self.emit(OpCode::PushLocalMult(ass, 1));
        self.emit(OpCode::PopLocal(state));

        let ctrl = vars[0];
        self.emit(OpCode::PushLocalMult(ass, 2));
        self.emit(OpCode::PopLocal(ctrl));

        // Start of loop
        let start = self.pos();

        // Call iterator
        self.emit(OpCode::PushLocal(ctrl));
        self.emit(OpCode::PushLocal(state));
        self.emit(OpCode::PushLocal(iter));
        self.emit(OpCode::Call(2));

        // Assign variables
        self.emit(OpCode::PopLocal(ass));
        for (ind, var) in vars.into_iter().enumerate() {
            self.emit(OpCode::PushLocalMult(ass, ind));
            self.emit(OpCode::PopLocal(var));
        }

        // Check control variable
        self.emit(OpCode::PushLocal(ctrl));
        self.emit(OpCode::PushLit(Value::Nil));
        self.emit(OpCode::BinOp(BinOp::Eq));

        // Jump to end if false, placeholder
        let jump_end = self.pos();
        self.emit(OpCode::Jump(0)); // Placeholder

        // Block
        self.compile_block(*r#for.block);

        // Jump back to start
        let off = offset(self.pos(), start);
        self.emit(OpCode::Jump(off));

        // Fill in jump to end
        let off = offset(jump_end, self.pos());
        self.set(jump_end, OpCode::JumpIf(off));

        self.scope_leave();
    }
}
