use crate::ast::{Exp, TableField};
use crate::compiler::{offset, Compiler};
use crate::vm::{BinOp, OpCode, Value};

impl Compiler {
    pub(super) fn compile_exp(&mut self, exp: Exp) {
        match exp {
            Exp::BinOp(lhs, op, rhs) => match op {
                // Short circuit logical operators
                BinOp::And | BinOp::Or => {
                    self.compile_exp(*lhs);
                    self.emit(OpCode::Copy);

                    let jump_pos = self.pos();
                    self.emit(OpCode::Jump(0)); // Placeholder

                    self.emit(OpCode::PopDisc);
                    self.compile_exp(*rhs);

                    let offset = offset(jump_pos, self.pos());
                    self.set(
                        jump_pos,
                        match op {
                            BinOp::And => OpCode::JumpIfNot(offset as i64),
                            BinOp::Or => OpCode::JumpIf(offset as i64),
                            _ => unreachable!(),
                        },
                    );
                }
                op => {
                    self.compile_exp(*lhs);
                    self.compile_exp(*rhs);
                    self.emit(OpCode::BinOp(op));
                }
            },
            Exp::UnOp(op, exp) => {
                self.compile_exp(*exp);
                self.emit(OpCode::UnOp(op));
            }
            Exp::Lit(val) => self.emit(OpCode::PushLit(val)),
            Exp::Prefix(exp) => self.compile_prefix_exp(exp),
            Exp::Table(fields) => {
                let len = fields.len();
                let mut index = 1;

                // TODO: Nil and NaN cannot be keys
                for field in fields {
                    match field {
                        TableField::Index(key, val) => {
                            self.compile_exp(val);
                            self.compile_exp(key);
                        }
                        TableField::List(val) => {
                            self.compile_exp(val);
                            self.emit(OpCode::PushLit(Value::int(index)));
                            index += 1;
                        }
                    }
                }
                self.emit(OpCode::BuildTbl(len));
            }
            Exp::Func(func) => self.compile_func(func),
            Exp::Varargs => {
                self.emit(OpCode::PushVarargs);
            }
        }
    }
}
