use crate::ast::{Exp, Parser};
use crate::error::Result;
use crate::token::Token;
use crate::vm::Value;

/// ```text
/// tableconstructor ::= ‘{’ [fieldlist] ‘}’
/// fieldlist ::= field {fieldsep field} [fieldsep]
/// field ::= ‘[’ exp ‘]’ ‘=’ exp | Name ‘=’ exp | exp
/// fieldsep ::= ‘,’ | ‘;’
/// ```
pub enum TableField {
    Index(Exp, Exp),
    List(Exp),
}

impl<'a> Parser<'a> {
    pub(super) fn parse_table_constructor(&mut self) -> Result<Vec<TableField>> {
        let mut fields = Vec::new();

        if !matches!(self.tokens.next(), Token::CurlO) {
            panic!()
        }

        // Consume first field (if there is any)
        if !matches!(self.tokens.peek(), Token::CurlC) {
            fields.push(self.parse_table_field()?);
        }
        loop {
            match self.tokens.next() {
                Token::CurlC => break,
                Token::Comma | Token::Semi => fields.push(self.parse_table_field()?),
                _ => panic!(),
            }
        }

        Ok(fields)
    }

    fn parse_table_field(&mut self) -> Result<TableField> {
        Ok(match self.tokens.peek() {
            Token::SqrO => {
                self.tokens.next();
                let index = self.parse_exp()?;
                if !matches!(self.tokens.next(), Token::SqrC) {
                    panic!()
                }
                if !matches!(self.tokens.next(), Token::Is) {
                    panic!()
                }
                TableField::Index(index, self.parse_exp()?)
            }
            Token::Ident(_) => {
                let name = match self.tokens.next() {
                    Token::Ident(name) => name.to_string(),
                    _ => unreachable!(),
                };
                if !matches!(self.tokens.next(), Token::Is) {
                    panic!()
                }
                TableField::Index(Exp::Lit(Value::String(name)), self.parse_exp()?)
            }
            _ => TableField::List(self.parse_exp()?),
        })
    }
}
