use crate::ast::{Ass, Block, Exp, For, Func, If, Local, Parser, PrefixExp, Repeat, While};
use crate::error::Result;
use crate::token::Token;

/// ```text
/// stat ::=  ‘;’ |
///      varlist ‘=’ explist |
///      functioncall |
///      label |
///      break |
///      goto Name |
///      do block end |
///      while exp do block end |
///      repeat block until exp |
///      if exp then block {elseif exp then block} [else block] end |
///      for Name ‘=’ exp ‘,’ exp [‘,’ exp] do block end |
///      for namelist in explist do block end |
///      function funcname funcbody |
///      local function Name funcbody |
///      local attnamelist [‘=’ explist]
/// label ::= ‘::’ Name ‘::’
/// ```
pub enum Stat {
    Nop,
    Ass(Ass),
    Call(PrefixExp, Vec<Exp>),
    CallMethod(Box<PrefixExp>, String, Vec<Exp>),
    Func(String, Func),
    LocalFunc(String, Func),
    Local(Local),
    Do(Block),
    If(If),
    Break,
    While(While),
    Repeat(Repeat),
    For(For),
    Goto(String),
    Label(String),
}

impl<'a> Parser<'a> {
    pub(super) fn parse_stat(&mut self) -> Result<Stat> {
        Ok(match self.tokens.peek() {
            Token::Semi => {
                self.tokens.next();
                Stat::Nop
            }
            Token::Ident(_) | Token::ParO => {
                let prefix_exp = self.parse_prefix_exp()?;

                match prefix_exp {
                    PrefixExp::Var(..) | PrefixExp::Index(..) => {
                        Stat::Ass(self.parse_ass(prefix_exp)?)
                    }
                    PrefixExp::Call(func, args) => Stat::Call(*func, args),
                    PrefixExp::CallMethod(table, name, args) => Stat::CallMethod(table, name, args),
                    _ => panic!(),
                }
            }
            Token::Local => {
                self.tokens.next();
                match self.tokens.peek() {
                    Token::Function => {
                        self.tokens.next();
                        let name = match self.tokens.next() {
                            Token::Ident(name) => name.to_string(),
                            _ => panic!(),
                        };

                        let func = self.parse_func()?;
                        Stat::LocalFunc(name, func)
                    }
                    _ => Stat::Local(self.parse_local()?),
                }
            }
            Token::Function => {
                self.tokens.next();
                let name = match self.tokens.next() {
                    Token::Ident(name) => name.to_string(),
                    _ => panic!(),
                };

                let func = self.parse_func()?;
                Stat::Func(name, func)
            }
            Token::Do => {
                self.tokens.next();
                let block = self.parse_block()?;
                if !matches!(self.tokens.next(), Token::End) {
                    panic!()
                }
                Stat::Do(block)
            }
            Token::If => Stat::If(self.parse_if()?),
            Token::Break => {
                self.tokens.next();
                Stat::Break
            }
            Token::While => Stat::While(self.parse_while()?),
            Token::Repeat => Stat::Repeat(self.parse_repeat()?),
            Token::For => Stat::For(self.parse_for()?),
            Token::Goto => {
                self.tokens.next();
                let label = match self.tokens.next() {
                    Token::Ident(name) => name.to_string(),
                    _ => panic!(),
                };
                Stat::Goto(label)
            }
            Token::Ass => {
                self.tokens.next();
                let label = match self.tokens.next() {
                    Token::Ident(name) => name.to_string(),
                    _ => panic!(),
                };
                if !matches!(self.tokens.next(), Token::Ass) {
                    panic!()
                }
                Stat::Label(label)
            }
            _ => panic!("{:?}", self.tokens.peek()),
        })
    }
}
