use crate::ast::{Exp, Parser};
use crate::error::Result;
use crate::token::Token;
use crate::vm::Value;

/// ```text
/// prefixexp ::= var | functioncall | ‘(’ exp ‘)’
/// var ::=  Name | prefixexp ‘[’ exp ‘]’ | prefixexp ‘.’ Name
/// functioncall ::=  prefixexp args | prefixexp ‘:’ Name args
/// args ::=  ‘(’ [explist] ‘)’ | tableconstructor | LiteralString
/// ```
///
/// This means that a prefix expression always starts with either a `Name` or `‘(’`
pub enum PrefixExp {
    Var(String),
    Index(Box<PrefixExp>, Box<Exp>),
    Call(Box<PrefixExp>, Vec<Exp>),
    CallMethod(Box<PrefixExp>, String, Vec<Exp>),
    Par(Box<Exp>),
}

impl<'a> Parser<'a> {
    pub(super) fn parse_prefix_exp(&mut self) -> Result<PrefixExp> {
        let mut res = match self.tokens.next() {
            Token::Ident(name) => PrefixExp::Var(name.to_string()),
            Token::ParO => {
                let res = PrefixExp::Par(Box::new(self.parse_exp()?));

                if !matches!(self.tokens.next(), Token::ParC) {
                    panic!()
                }

                res
            }
            _ => panic!(),
        };

        loop {
            match self.tokens.peek() {
                Token::SqrO => {
                    self.tokens.next();

                    let exp = self.parse_exp()?;

                    if !matches!(self.tokens.next(), Token::SqrC) {
                        panic!()
                    }

                    res = PrefixExp::Index(Box::new(res), Box::new(exp));
                }
                Token::Point => {
                    self.tokens.next();

                    if let Token::Ident(name) = self.tokens.next() {
                        res = PrefixExp::Index(
                            Box::new(res),
                            Box::new(Exp::Lit(Value::String(name.to_string()))),
                        );
                    } else {
                        panic!()
                    }
                }
                Token::ParO | Token::CurlO | Token::String(..) => {
                    let args = self.parse_args()?;

                    res = PrefixExp::Call(Box::new(res), args);
                }
                Token::Colon => {
                    self.tokens.next();

                    let name = match self.tokens.next() {
                        Token::Ident(name) => name.to_string(),
                        _ => panic!(),
                    };
                    let args = self.parse_args()?;

                    res = PrefixExp::CallMethod(Box::new(res), name, args);
                }
                _ => break,
            }
        }

        Ok(res)
    }

    fn parse_args(&mut self) -> Result<Vec<Exp>> {
        match self.tokens.peek() {
            Token::ParO => {
                self.tokens.next();

                let mut args = Vec::new();

                loop {
                    match self.tokens.peek() {
                        Token::ParC => break,
                        Token::Comma => {
                            if args.is_empty() {
                                panic!()
                            } else {
                                self.tokens.next();
                            }
                        }
                        _ => {}
                    }

                    args.push(self.parse_exp()?);
                }

                if !matches!(self.tokens.next(), Token::ParC) {
                    panic!()
                }

                Ok(args)
            }
            Token::CurlO => {
                let fields = self.parse_table_constructor()?;
                let exp = Exp::Table(fields);

                Ok(vec![exp])
            }
            Token::String(str) => {
                let exp = Exp::Lit(Value::String(str.to_string()));
                self.tokens.next();

                Ok(vec![exp])
            }
            _ => panic!(),
        }
    }
}
