use crate::ast::{Exp, Parser};
use crate::error::Result;
use crate::token::Token;

/// ```text
/// stat ::= local function Name funcbody |
///          local attnamelist [‘=’ explist]
/// attnamelist ::=  Name attrib {‘,’ Name attrib}
/// ```
pub struct Local {
    pub vars: Vec<(String, Option<Attr>)>,
    pub exps: Vec<Exp>,
}

#[derive(Clone, Copy)]
pub enum Attr {
    Const,
    Close,
}

impl<'a> Parser<'a> {
    pub(super) fn parse_local(&mut self) -> Result<Local> {
        let mut vars = vec![self.parse_name_attr()?];
        loop {
            match self.tokens.peek() {
                Token::Comma => {
                    self.tokens.next();
                    vars.push(self.parse_name_attr()?);
                }
                Token::Is => {
                    self.tokens.next();
                    break;
                }
                _ => {
                    return Ok(Local {
                        vars,
                        exps: Vec::new(),
                    })
                }
            }
        }

        let mut exps = vec![self.parse_exp()?];
        while let Token::Comma = self.tokens.peek() {
            self.tokens.next();
            exps.push(self.parse_exp()?);
        }

        Ok(Local { vars, exps })
    }

    fn parse_name_attr(&mut self) -> Result<(String, Option<Attr>)> {
        let name = match self.tokens.next() {
            Token::Ident(name) => name.to_string(),
            _ => panic!(),
        };
        let attr = match self.tokens.peek() {
            Token::Lt => {
                self.tokens.next();
                let attr = match self.tokens.next() {
                    Token::Ident(name) => match name.as_str() {
                        "const" => Attr::Const,
                        "close" => Attr::Close,
                        _ => panic!(),
                    },
                    _ => panic!(),
                };
                if !matches!(self.tokens.next(), Token::Gt) {
                    panic!()
                }
                Some(attr)
            }
            _ => None,
        };
        Ok((name, attr))
    }
}
