use crate::ast::{Block, Exp, Parser};
use crate::error::Result;
use crate::token::Token;

/// ```text
/// stat ::= if exp then block {elseif exp then block} [else block] end
/// ```
pub struct If {
    pub conds: Vec<(Exp, Block)>,
    pub els: Option<Box<Block>>,
}

impl<'a> Parser<'a> {
    pub(super) fn parse_if(&mut self) -> Result<If> {
        if !matches!(self.tokens.next(), Token::If) {
            panic!()
        }

        let mut conds = vec![self.parse_cond()?];
        let mut els = None;
        loop {
            match self.tokens.next() {
                Token::Elseif => conds.push(self.parse_cond()?),
                Token::Else => {
                    els = Some(Box::new(self.parse_block()?));
                    if !matches!(self.tokens.next(), Token::End) {
                        panic!()
                    }
                    break;
                }
                Token::End => break,
                _ => panic!(),
            }
        }

        Ok(If { conds, els })
    }

    fn parse_cond(&mut self) -> Result<(Exp, Block)> {
        let exp = self.parse_exp()?;
        if !matches!(self.tokens.next(), Token::Then) {
            panic!()
        }
        let block = self.parse_block()?;
        Ok((exp, block))
    }
}
