use crate::ast::{Block, Parser};
use crate::error::Result;
use crate::token::Token;

/// ```text
/// stat ::= function funcname funcbody
/// functiondef ::= function funcbody
/// funcbody ::= ‘(’ [parlist] ‘)’ block end
/// parlist ::= namelist [‘,’ ‘...’] | ‘...’
/// ```
pub struct Func {
    pub params: Vec<String>,
    pub varargs: bool,
    pub block: Box<Block>,
}

impl<'a> Parser<'a> {
    pub(super) fn parse_func(&mut self) -> Result<Func> {
        let (params, varargs) = self.parse_params()?;

        let block = self.parse_block()?;

        if !matches!(self.tokens.next(), Token::End) {
            panic!()
        }

        Ok(Func {
            params,
            varargs,
            block: Box::new(block),
        })
    }

    fn parse_params(&mut self) -> Result<(Vec<String>, bool)> {
        if !matches!(self.tokens.next(), Token::ParO) {
            panic!()
        }

        let mut params = Vec::new();
        let mut varargs = false;

        if let Token::ParC = self.tokens.peek() {
            self.tokens.next();
            return Ok((params, varargs));
        }

        loop {
            match self.tokens.next() {
                Token::Ident(name) => {
                    params.push(name.to_string());
                    match self.tokens.next() {
                        Token::Comma => {}
                        Token::ParC => break,
                        _ => panic!(),
                    }
                }
                Token::Dots => {
                    varargs = true;
                    if !matches!(self.tokens.next(), Token::ParC) {
                        panic!()
                    }
                    break;
                }
                Token::ParC => break,
                _ => panic!(),
            }
        }

        Ok((params, varargs))
    }
}
