use crate::ast::{Block, Exp, Parser};
use crate::error::Result;
use crate::token::Token;

/// ```text
/// stat ::= for Name ‘=’ exp ‘,’ exp [‘,’ exp] do block end |
///          for namelist in explist do block end
/// namelist ::= Name {‘,’ Name}
/// explist ::= exp {‘,’ exp}
/// ```

pub enum For {
    Numeric(ForNum),
    Generic(ForGen),
}

pub struct ForNum {
    pub name: String,
    pub init: Box<Exp>,
    pub limit: Box<Exp>,
    pub step: Option<Box<Exp>>,
    pub block: Box<Block>,
}

pub struct ForGen {
    pub names: Vec<String>,
    pub exp: Box<Exp>,
    pub block: Box<Block>,
}

impl<'a> Parser<'a> {
    pub(super) fn parse_for(&mut self) -> Result<For> {
        if !matches!(self.tokens.next(), Token::For) {
            panic!()
        }

        let name = match self.tokens.next() {
            Token::Ident(name) => name.to_string(),
            _ => panic!(),
        };

        Ok(match self.tokens.peek() {
            Token::Is => For::Numeric(self.parse_for_num(name)?),
            Token::Comma | Token::In => For::Generic(self.parse_for_gen(name)?),
            _ => panic!(),
        })
    }

    fn parse_for_num(&mut self, name: String) -> Result<ForNum> {
        if !matches!(self.tokens.next(), Token::Is) {
            panic!()
        }

        let init = Box::new(self.parse_exp()?);

        if !matches!(self.tokens.next(), Token::Comma) {
            panic!()
        }

        let limit = Box::new(self.parse_exp()?);

        let step = match self.tokens.peek() {
            Token::Comma => {
                self.tokens.next();
                Some(Box::new(self.parse_exp()?))
            }
            _ => None,
        };

        if !matches!(self.tokens.next(), Token::Do) {
            panic!()
        }

        let block = Box::new(self.parse_block()?);

        if !matches!(self.tokens.next(), Token::End) {
            panic!()
        }

        Ok(ForNum {
            name,
            init,
            limit,
            step,
            block,
        })
    }

    fn parse_for_gen(&mut self, name: String) -> Result<ForGen> {
        let mut names = vec![name];

        loop {
            match self.tokens.next() {
                Token::Comma => match self.tokens.next() {
                    Token::Ident(name) => names.push(name.to_string()),
                    _ => panic!(),
                },
                Token::In => break,
                _ => panic!(),
            }
        }

        let exp = Box::new(self.parse_exp()?);

        if !matches!(self.tokens.next(), Token::Do) {
            panic!()
        }

        let block = Box::new(self.parse_block()?);

        if !matches!(self.tokens.next(), Token::End) {
            panic!()
        }

        Ok(ForGen { names, exp, block })
    }
}
