use crate::ast::{Func, Parser, PrefixExp, TableField};
use crate::error::Result;
use crate::token::Token;
use crate::vm::{BinOp, UnOp, Value};

/// ```text
/// exp ::=  nil | false | true | Numeral | LiteralString | ‘...’ | functiondef |
///          prefixexp | tableconstructor | exp binop exp | unop exp
/// binop ::=  ‘+’ | ‘-’ | ‘*’ | ‘/’ | ‘//’ | ‘^’ | ‘%’ |
///            ‘&’ | ‘~’ | ‘|’ | ‘>>’ | ‘<<’ | ‘..’ |
///            ‘<’ | ‘<=’ | ‘>’ | ‘>=’ | ‘==’ | ‘~=’ |
///            and | or
/// unop ::= ‘-’ | not | ‘#’ | ‘~’
/// ```
pub enum Exp {
    BinOp(Box<Exp>, BinOp, Box<Exp>),
    UnOp(UnOp, Box<Exp>),
    Lit(Value),
    Table(Vec<TableField>),
    Prefix(PrefixExp),
    Func(Func),
    Varargs,
}

// Indicating what was previously encountered
#[derive(PartialEq)]
enum ExpParseState {
    Init,
    Operand,
    Operator,
    Done,
}

enum Op {
    BinOp(BinOp),
    UnOp(UnOp),
}

struct ExpParser<'a, 'b> {
    parser: &'a mut Parser<'b>,
    state: ExpParseState,
    out: Vec<Exp>,
    ops: Vec<Op>,
}

impl<'a> Parser<'a> {
    pub(super) fn parse_exp(&mut self) -> Result<Exp> {
        ExpParser {
            parser: self,
            state: ExpParseState::Init,
            out: Vec::new(),
            ops: Vec::new(),
        }
        .parse()
    }
}

impl Op {
    fn precedence(&self) -> usize {
        match self {
            Op::BinOp(op) => op.precedence(),
            Op::UnOp(op) => op.precedence(),
        }
    }
}

impl<'a, 'b> ExpParser<'a, 'b> {
    fn parse(mut self) -> Result<Exp> {
        while self.state != ExpParseState::Done {
            self.transition()?;
        }

        self.finalize()
    }

    fn transition(&mut self) -> Result<()> {
        self.state = match self.state {
            ExpParseState::Init => {
                // Expecting operand or unary operator
                if let Some(op) = self.peek_un_op() {
                    self.parser.tokens.next();
                    self.ops.push(Op::UnOp(op));

                    ExpParseState::Operator
                } else {
                    let op = self.parse_operand()?;
                    self.out.push(op);

                    ExpParseState::Operand
                }
            }
            ExpParseState::Operand => {
                // Expecting binary operator or end
                if let Some(op) = self.peek_bin_op() {
                    while let Some(pop) = self.ops.last() {
                        if pop.precedence() > op.precedence()
                            || (op.is_left_ass() && pop.precedence() == op.precedence())
                        {
                            self.pop_op()?;
                        } else {
                            break;
                        }
                    }

                    self.parser.tokens.next();
                    self.ops.push(Op::BinOp(op));

                    ExpParseState::Operator
                } else {
                    ExpParseState::Done
                }
            }
            ExpParseState::Operator => {
                // Expecting operand or unary operator
                if let Some(op) = self.peek_un_op() {
                    self.parser.tokens.next();
                    self.ops.push(Op::UnOp(op));

                    ExpParseState::Operator
                } else {
                    let op = self.parse_operand()?;
                    self.out.push(op);

                    ExpParseState::Operand
                }
            }
            ExpParseState::Done => unreachable!(),
        };

        Ok(())
    }

    fn finalize(mut self) -> Result<Exp> {
        while !self.ops.is_empty() {
            self.pop_op()?;
        }

        if self.out.len() == 1 {
            Ok(self.out.pop().unwrap())
        } else {
            panic!()
        }
    }

    fn parse_operand(&mut self) -> Result<Exp> {
        if self.peek_prefix_exp() {
            Ok(Exp::Prefix(self.parser.parse_prefix_exp()?))
        } else if let Some(lit) = self.peek_literal() {
            self.parser.tokens.next();
            Ok(Exp::Lit(lit))
        } else if self.peek_table() {
            Ok(Exp::Table(self.parser.parse_table_constructor()?))
        } else if self.peek_function() {
            self.parser.tokens.next();
            Ok(Exp::Func(self.parser.parse_func()?))
        } else if let Token::Dots = self.parser.tokens.peek() {
            self.parser.tokens.next();
            Ok(Exp::Varargs)
        } else {
            panic!("{:?}", self.parser.tokens.peek());
        }
    }

    fn peek_prefix_exp(&self) -> bool {
        matches!(self.parser.tokens.peek(), Token::Ident(..) | Token::ParO)
    }

    fn peek_table(&self) -> bool {
        matches!(self.parser.tokens.peek(), Token::CurlO)
    }

    fn peek_function(&self) -> bool {
        matches!(self.parser.tokens.peek(), Token::Function)
    }

    fn peek_literal(&self) -> Option<Value> {
        match self.parser.tokens.peek() {
            Token::Nil => Some(Value::Nil),
            Token::True => Some(Value::Bool(true)),
            Token::False => Some(Value::Bool(false)),
            Token::Integer(n) => Some(Value::int(*n)),
            Token::Float(f) => Some(Value::float(*f)),
            Token::String(s) => Some(Value::String(s.clone())),
            _ => None,
        }
    }

    fn peek_bin_op(&self) -> Option<BinOp> {
        match self.parser.tokens.peek() {
            // Mathematical
            Token::Add => Some(BinOp::Add),
            Token::Sub => Some(BinOp::Sub),
            Token::Mul => Some(BinOp::Mul),
            Token::Div => Some(BinOp::Div),
            Token::FlDiv => Some(BinOp::FlDiv),
            Token::Pow => Some(BinOp::Pow),
            Token::Mod => Some(BinOp::Mod),
            // Bitwise
            Token::Amp => Some(BinOp::BitAnd),
            Token::Tilde => Some(BinOp::BitXor),
            Token::Bar => Some(BinOp::BitOr),
            Token::Shl => Some(BinOp::Shl),
            Token::Shr => Some(BinOp::Shr),
            // String
            Token::Conc => Some(BinOp::Conc),
            // Comparison
            Token::Eq => Some(BinOp::Eq),
            Token::Neq => Some(BinOp::Neq),
            Token::Lt => Some(BinOp::Lt),
            Token::Leq => Some(BinOp::Leq),
            Token::Gt => Some(BinOp::Gt),
            Token::Geq => Some(BinOp::Geq),
            Token::And => Some(BinOp::And),
            Token::Or => Some(BinOp::Or),
            _ => None,
        }
    }

    fn peek_un_op(&self) -> Option<UnOp> {
        match self.parser.tokens.peek() {
            Token::Sub => Some(UnOp::Minus),
            Token::Tilde => Some(UnOp::BitNot),
            Token::Not => Some(UnOp::Not),
            Token::Hash => Some(UnOp::Len),
            _ => None,
        }
    }

    fn pop_op(&mut self) -> Result<()> {
        let op = match self.ops.pop() {
            Some(o) => o,
            None => panic!(),
        };

        let out = match op {
            Op::BinOp(op) => {
                let op2 = match self.out.pop() {
                    Some(v) => v,
                    None => panic!(),
                };
                let op1 = match self.out.pop() {
                    Some(v) => v,
                    None => panic!(),
                };

                Exp::BinOp(Box::new(op1), op, Box::new(op2))
            }
            Op::UnOp(op) => {
                let op1 = match self.out.pop() {
                    Some(v) => v,
                    None => panic!(),
                };

                Exp::UnOp(op, Box::new(op1))
            }
        };
        self.out.push(out);

        Ok(())
    }
}
