use crate::ast::{Exp, Parser, PrefixExp};
use crate::error::Result;
use crate::token::Token;

/// ```text
/// stat ::= varlist ‘=’ explist
/// varlist ::= var {‘,’ var}
/// explist ::= exp {‘,’ exp}
/// ```
pub struct Ass {
    pub vars: Vec<Var>,
    pub exps: Vec<Exp>,
}

pub enum Var {
    Name(String),
    Index(PrefixExp, Exp),
}

impl<'a> Parser<'a> {
    pub(super) fn parse_ass(&mut self, first: PrefixExp) -> Result<Ass> {
        let mut vars = vec![self.parse_var(Some(first))?];
        loop {
            match self.tokens.next() {
                Token::Comma => vars.push(self.parse_var(None)?),
                Token::Is => break,
                _ => panic!(),
            }
        }

        let mut exps = vec![self.parse_exp()?];
        while let Token::Comma = self.tokens.peek() {
            self.tokens.next();
            exps.push(self.parse_exp()?);
        }

        Ok(Ass { vars, exps })
    }

    fn parse_var(&mut self, parsed: Option<PrefixExp>) -> Result<Var> {
        let prefix_exp = match parsed {
            Some(exp) => exp,
            None => self.parse_prefix_exp()?,
        };

        Ok(match prefix_exp {
            PrefixExp::Var(name) => Var::Name(name),
            PrefixExp::Index(sub, exp) => Var::Index(*sub, *exp),
            _ => panic!(),
        })
    }
}
