use crate::prelude::*;

#[derive(Eq, PartialEq, Debug, Clone)]
pub enum ConvOp {
    Bytes,
    Duration,
    DurationSeconds,
}

impl Parser for ConvOp {
    fn parse(input: &str) -> IResult<&str, Self> {
        alt((
            map(tag("bytes"), |_| ConvOp::Bytes),
            map(tag("duration"), |_| ConvOp::Duration),
            map(tag("duration_seconds"), |_| ConvOp::DurationSeconds),
        ))(input)
    }
}

// RangeOp
// range vector aggregation
#[derive(Eq, PartialEq, Debug, Clone)]
pub enum RangeOp {
    Count,
    Rate,
    Bytes,
    BytesRate,
    Avg,
    Sum,
    Min,
    Max,
    Stdvar,
    Stddev,
    Quantile,
    First,
    Last,
    Absent,
}

impl Parser for RangeOp {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("RangeOp {}", input);
        alt((
            map(tag("count_over_time"), |_| RangeOp::Count),
            map(tag("rate"), |_| RangeOp::Rate),
            map(tag("bytes_over_time"), |_| RangeOp::Bytes),
            map(tag("bytes_rate"), |_| RangeOp::BytesRate),
            map(tag("avg_over_time"), |_| RangeOp::Avg),
            map(tag("sum_over_time"), |_| RangeOp::Sum),
            map(tag("min_over_time"), |_| RangeOp::Min),
            map(tag("max_over_time"), |_| RangeOp::Max),
            map(tag("stdvar_over_time"), |_| RangeOp::Stdvar),
            map(tag("stddev_over_time"), |_| RangeOp::Stddev),
            map(tag("quantile_over_time"), |_| RangeOp::Quantile),
            map(tag("first_over_time"), |_| RangeOp::First),
            map(tag("last_over_time"), |_| RangeOp::Last),
            map(tag("absent_over_time"), |_| RangeOp::Absent),
        ))(input)
    }
}

// MatchOp      ::= '=' | '!=' | '=~' | '!~'
// used for log stream selector to filter labels
// e.g. {name=~"mysql.+", env="prod"}
#[derive(Eq, PartialEq, Debug, Copy, Clone)]
pub enum MatchOp {
    // match
    EQ,
    // not match
    NEQ,
    // regex match
    RE,
    // regex not match
    NRE,
}

impl Parser for MatchOp {
    // note: alt tries to parse in order, so we should try RE first
    //      otherwise, EQ will be returned in EQ & RE case
    fn parse(input: &str) -> IResult<&str, Self> {
        alt((
            map(tag("=~"), |_| Self::RE),
            map(tag("="), |_| Self::EQ),
            map(tag("!="), |_| Self::NEQ),
            map(tag("!~"), |_| Self::NRE),
        ))(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone, Copy)]
pub enum LabelFilterOp {
    AND,
    OR,
}

impl Parser for LabelFilterOp {
    fn parse(input: &str) -> IResult<&str, Self> {
        alt((
            map(tag("and"), |_| LabelFilterOp::AND),
            map(tag("or"), |_| LabelFilterOp::OR),
        ))(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub enum LabelFilterType {
    EQ,
    NEQ,
    GT,
    GTE,
    LT,
    LTE,
}

impl Parser for LabelFilterType {
    fn parse(input: &str) -> IResult<&str, Self> {
        alt((
            map(tag("=="), |_| LabelFilterType::EQ),
            map(tag("="), |_| LabelFilterType::EQ),
            map(tag("!="), |_| LabelFilterType::NEQ),
            map(tag(">="), |_| LabelFilterType::GTE),
            map(tag(">"), |_| LabelFilterType::GT),
            map(tag("<="), |_| LabelFilterType::LTE),
            map(tag("<"), |_| LabelFilterType::LT),
        ))(input)
    }
}

// Filter       ::= '|=' | '|~' '!~' | '!='
// e.g. {job="mysql"} |= "error" != "timeout"
#[derive(Eq, PartialEq, Debug, Clone)]
pub enum Filter {
    // log line match regex
    PipeMatch,
    // log line contains
    PipeExact,
    // log line not contains
    NRE,
    // log line not match regex
    NEQ,
}

impl Parser for Filter {
    fn parse(input: &str) -> IResult<&str, Self> {
        alt((
            map(tag("|~"), |_| Self::PipeMatch),
            map(tag("|="), |_| Self::PipeExact),
            map(tag("!="), |_| Self::NEQ),
            map(tag("!~"), |_| Self::NRE),
        ))(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub enum BinOp {
    OR,
    AND,
    UNLESS,
    ADD,
    SUB,
    MUL,
    DIV,
    MOD,
    POW,
    CmpEq,
    NEQ,
    GT,
    GTE,
    LT,
    LTE,
}

impl Parser for BinOp {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("BinOp {}", input);
        alt((
            map(tag("or"), |_| Self::OR),
            map(tag("and"), |_| Self::AND),
            map(tag("unless"), |_| Self::UNLESS),
            map(tag("+"), |_| Self::ADD),
            map(tag("-"), |_| Self::SUB),
            map(tag("*"), |_| Self::MUL),
            map(tag("/"), |_| Self::DIV),
            map(tag("%"), |_| Self::MOD),
            map(tag("^"), |_| Self::POW),
            map(tag("=="), |_| Self::CmpEq),
            map(tag("!="), |_| Self::NEQ),
            map(tag(">"), |_| Self::GT),
            map(tag(">="), |_| Self::GTE),
            map(tag("<"), |_| Self::LT),
            map(tag("<="), |_| Self::LTE),
        ))(input)
    }
}

// VectorOp
// aggregation operators
#[derive(Eq, PartialEq, Debug, Clone)]
pub enum VectorOp {
    SUM,
    AVG,
    COUNT,
    MAX,
    MIN,
    STDDEV,
    STDVAR,
    BOTTOMK,
    TOPK,
}

impl Parser for VectorOp {
    fn parse(input: &str) -> IResult<&str, Self> {
        alt((
            map(tag("sum"), |_| VectorOp::SUM),
            map(tag("avg"), |_| VectorOp::AVG),
            map(tag("count"), |_| VectorOp::COUNT),
            map(tag("max"), |_| VectorOp::MAX),
            map(tag("min"), |_| VectorOp::MIN),
            map(tag("stddev"), |_| VectorOp::STDDEV),
            map(tag("stdvar"), |_| VectorOp::STDVAR),
            map(tag("bottomk"), |_| VectorOp::BOTTOMK),
            map(tag("topk"), |_| VectorOp::TOPK),
        ))(input)
    }
}
mod tests {
    #[test]
    fn test_match_op() {
        assert_eq!(MatchOp::parse("=").unwrap().1, MatchOp::EQ);
        assert_eq!(MatchOp::parse("!=").unwrap().1, MatchOp::NEQ);
        assert_eq!(MatchOp::parse("=~").unwrap().1, MatchOp::RE);
        assert_eq!(MatchOp::parse("!~").unwrap().1, MatchOp::NRE);
    }
}
