use crate::{prelude::*, Identifier, Ip, LiteralExpr, Separator, Value};
use crate::{BytesUnit, DurationUnit, Filter, LabelFilterOp, LabelFilterType, MatchOp};

#[derive(PartialEq, Debug, Clone)]
pub enum LogExpr {
    MatcherExpr(Selector),
    PipelineExpr(Selector, PipelineExpr),
}

impl Parser for LogExpr {
    fn parse(input: &str) -> IResult<&str, Self> {
        alt((
            map(
                tuple((
                    Selector::parse,
                    opt(Separator::parse),
                    opt(PipelineExpr::parse),
                )),
                |(selector, _, pipeline_expr)| match pipeline_expr {
                    Some(pipeline_expr) => Self::PipelineExpr(selector, pipeline_expr),
                    None => Self::MatcherExpr(selector),
                },
            ),
            delimited(cchar('('), Self::parse, cchar(')')),
        ))(input)
    }
}

#[derive(From, Eq, PartialEq, Debug, Clone)]
pub struct Selector(Option<Matchers>);

impl Parser for Selector {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            tuple((
                tag("{"),
                opt(Separator::parse),
                alt((
                    map(
                        tuple((
                            Matchers::parse,
                            pair(opt(Separator::parse), alt((tag("error"), tag("}")))),
                        )),
                        |(matchers, _)| Self(Some(matchers)),
                    ),
                    map(tuple((Matchers::parse, tag("error"))), |_abc1| Self(None)),
                )),
                opt(Separator::parse),
            )),
            |(_, _, selector, _)| selector,
        )(input)
    }
}

#[derive(PartialEq, Debug, Clone)]
pub struct PipelineExpr(Vec<PipelineStage>);

impl Parser for PipelineExpr {
    fn parse(input: &str) -> IResult<&str, Self> {
        let mut stages = Vec::new();
        let (remains, _) = many1(map(
            tuple((PipelineStage::parse, opt(Separator::parse))),
            |(i, _)| stages.push(i),
        ))(input)?;
        Ok((remains, Self(stages)))
    }
}

#[derive(PartialEq, Debug, Clone)]
pub enum PipelineStage {
    LineFilters(LineFilters),
    LabelParser(LabelParserExpr),
    JsonExpressionParser(JsonExpressionParser),
    LabelFilters(LabelFilters),
    LineFormatExpr(LineFormatExpr),
    LabelFormatExpr(LabelFormatExpr),
}

impl Parser for PipelineStage {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("PipelineStage {}", input);
        alt((
            map(LineFilters::parse, PipelineStage::LineFilters),
            map(
                tuple((
                    tag("|"),
                    opt(Separator::parse),
                    alt((
                        map(
                            JsonExpressionParser::parse,
                            PipelineStage::JsonExpressionParser,
                        ),
                        map(LabelParserExpr::parse, PipelineStage::LabelParser),
                        map(LabelFilters::parse, PipelineStage::LabelFilters),
                        map(LineFormatExpr::parse, PipelineStage::LineFormatExpr),
                        map(LabelFormatExpr::parse, PipelineStage::LabelFormatExpr),
                    )),
                )),
                |(_, _, i)| i,
            ),
        ))(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct JsonExpressionParser(JsonExpressionList);

impl Parser for JsonExpressionParser {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("JsonExpressionParser {}", input);
        map(
            tuple((
                tag("json"),
                opt(Separator::parse),
                JsonExpressionList::parse,
                opt(Separator::parse),
            )),
            |(_, _, i, _)| JsonExpressionParser(i),
        )(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct JsonExpressionList(Vec<JsonExpression>);

impl Parser for JsonExpressionList {
    fn parse(input: &str) -> IResult<&str, Self> {
        let (remains, jsons) = separated_list1(
            tag(","),
            delimited(
                opt(Separator::parse),
                JsonExpression::parse,
                opt(Separator::parse),
            ),
        )(input)?;
        Ok((remains, Self(jsons)))
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct JsonExpression((Identifier, Value));

impl Parser for JsonExpression {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            tuple((
                Identifier::parse,
                opt(Separator::parse),
                tag("="),
                opt(Separator::parse),
                Value::parse,
                opt(Separator::parse),
            )),
            |(key, _, _, _, val, _)| JsonExpression((key, val)),
        )(input)
    }
}

#[derive(PartialEq, Debug, Clone)]
pub enum LabelFilters {
    Single(LabelFilter),
    Compose {
        op: LabelFilterOp,
        lhs: Box<LabelFilters>,
        rhs: Box<LabelFilters>,
    },
}

impl Parser for LabelFilters {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("LabelFilters {}", input);
        alt((
            delimited(
                cchar('('),
                delimited(
                    opt(Separator::parse),
                    LabelFilters::parse,
                    opt(Separator::parse),
                ),
                cchar(')'),
            ),
            map(
                tuple((
                    LabelFilter::parse,
                    opt(alt((
                        map(tuple((opt(Separator::parse), tag("and"))), |_| {
                            LabelFilterOp::AND
                        }),
                        map(tuple((opt(Separator::parse), tag(","))), |_| {
                            LabelFilterOp::AND
                        }),
                        map(tuple((opt(Separator::parse), tag("or"))), |_| {
                            LabelFilterOp::OR
                        }),
                        map(multispace1, |_| LabelFilterOp::AND),
                    ))),
                    opt(Separator::parse),
                    opt(LabelFilters::parse),
                )),
                |(lhs, op, _, rhs)| {
                    let op = op.unwrap_or(LabelFilterOp::AND);
                    if let Some(rhs) = rhs {
                        LabelFilters::Compose {
                            op,
                            lhs: Box::new(LabelFilters::Single(lhs)),
                            rhs: Box::new(rhs),
                        }
                    } else {
                        LabelFilters::Single(lhs)
                    }
                },
            ),
        ))(input)
    }
}

#[derive(PartialEq, Debug, Clone)]
pub enum LabelFilter {
    Matcher(Matcher),
    Unit(UnitFilter),
    Number(NumberFilter<LiteralExpr>),
    IpTable(IpTableFilter),
}

impl Parser for LabelFilter {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            alt((
                map(Matcher::parse, LabelFilter::Matcher),
                map(UnitFilter::parse, LabelFilter::Unit),
                map(NumberFilter::<LiteralExpr>::parse, LabelFilter::Number),
                map(IpTableFilter::parse, LabelFilter::IpTable),
            )),
            |i| i,
        )(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct IpTableFilter {
    key: Identifier,
    val: Ip,
    op: MatchOp,
}

impl Parser for IpTableFilter {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            tuple((
                Identifier::parse,
                opt(Separator::parse),
                alt((
                    map(tag("="), |_| MatchOp::EQ),
                    map(tag("!="), |_| MatchOp::NEQ),
                )),
                opt(Separator::parse),
                tag("ip"),
                opt(Separator::parse),
                cchar('('),
                opt(Separator::parse),
                Ip::parse,
                opt(Separator::parse),
                cchar(')'),
            )),
            |(key, _, op, _, _, _, _, _, val, _, _)| Self { key, op, val },
        )(input)
    }
}

#[derive(PartialEq, Debug, Clone)]
pub enum UnitFilter {
    Duration(DurationFilter),
    Bytes(BytesFilter),
}

impl Parser for UnitFilter {
    fn parse(input: &str) -> IResult<&str, Self> {
        enum UnitFilterType {
            Duration(DurationUnit),
            Bytes(BytesUnit),
        }

        map(
            tuple((
                NumberFilter::<LiteralExpr>::parse,
                alt((
                    map(DurationUnit::parse, UnitFilterType::Duration),
                    map(BytesUnit::parse, UnitFilterType::Bytes),
                )),
            )),
            |(inner, unit)| match unit {
                UnitFilterType::Duration(unit) => UnitFilter::Duration(DurationFilter {
                    inner: NumberFilter {
                        op: inner.op,
                        key: inner.key,
                        val: (inner.val, unit),
                    },
                }),
                UnitFilterType::Bytes(unit) => UnitFilter::Bytes(BytesFilter {
                    inner: NumberFilter {
                        op: inner.op,
                        key: inner.key,
                        val: (inner.val, unit),
                    },
                }),
            },
        )(input)
    }
}

#[derive(PartialEq, Debug, Clone)]
pub struct DurationFilter {
    inner: NumberFilter<(LiteralExpr, DurationUnit)>,
}

impl Parser for DurationFilter {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            NumberFilter::<(LiteralExpr, DurationUnit)>::parse,
            |inner| Self { inner },
        )(input)
    }
}

#[derive(PartialEq, Debug, Clone)]
pub struct BytesFilter {
    inner: NumberFilter<(LiteralExpr, BytesUnit)>,
}

impl Parser for BytesFilter {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(NumberFilter::<(LiteralExpr, BytesUnit)>::parse, |inner| {
            Self { inner }
        })(input)
    }
}

#[derive(PartialEq, Debug, Clone)]
pub struct NumberFilter<T> {
    op: LabelFilterType,
    key: Identifier,
    val: T,
}

impl<T> Parser for NumberFilter<T>
where
    T: Parser,
{
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            tuple((
                Identifier::parse,
                opt(Separator::parse),
                LabelFilterType::parse,
                opt(Separator::parse),
                T::parse,
            )),
            |(key, _, op, _, val)| NumberFilter { op, key, val },
        )(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct LineFormatExpr(Value);

impl Parser for LineFormatExpr {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("LineFormatExpr {}", input);
        map(
            tuple((tag("line_format"), Separator::parse, Value::parse)),
            |(_, _, value)| LineFormatExpr(value),
        )(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct LabelFormatExpr(LabelsFormat);

impl Parser for LabelFormatExpr {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("LabelFormatExpr {}", input);
        map(LabelsFormat::parse, Self)(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct LabelsFormat(Vec<LabelFormat>);

impl Parser for LabelsFormat {
    fn parse(input: &str) -> IResult<&str, Self> {
        let (remains, labels) = separated_list1(
            tag(","),
            delimited(
                opt(Separator::parse),
                LabelFormat::parse,
                opt(Separator::parse),
            ),
        )(input)?;
        Ok((remains, Self(labels)))
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub enum LabelFormat {
    Rename((Identifier, Identifier)),
    Template((Identifier, Value)),
}

impl Parser for LabelFormat {
    fn parse(input: &str) -> IResult<&str, Self> {
        enum LabelFormatType {
            Identifier(Identifier),
            Value(Value),
        }

        map(
            tuple((
                Identifier::parse,
                opt(Separator::parse),
                tag("="),
                opt(Separator::parse),
                alt((
                    map(Identifier::parse, LabelFormatType::Identifier),
                    map(Value::parse, LabelFormatType::Value),
                )),
            )),
            |(identifier, _, _, _, label)| match label {
                LabelFormatType::Identifier(i) => LabelFormat::Rename((identifier, i)),
                LabelFormatType::Value(i) => LabelFormat::Template((identifier, i)),
            },
        )(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct LineFilters(Vec<LineFilter>);

impl Parser for LineFilters {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("LineFilters {}", input);
        let mut filters = Vec::new();
        let (remains, _) = many1(map(
            tuple((LineFilter::parse, opt(Separator::parse))),
            |(i, _)| filters.push(i),
        ))(input)?;
        Ok((remains, Self(filters)))
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct LineFilter {
    filter: Filter,
    filter_ip: bool,
    val: Value,
}

impl Parser for LineFilter {
    fn parse(input: &str) -> IResult<&str, Self> {
        enum LineFilterType {
            Normal(Value),
            Ip(Value),
        }
        map(
            tuple((
                Filter::parse,
                opt(Separator::parse),
                alt((
                    map(Value::parse, LineFilterType::Normal),
                    map(
                        tuple((
                            tag("ip"),
                            opt(Separator::parse),
                            delimited(
                                cchar('('),
                                delimited(
                                    opt(Separator::parse),
                                    Value::parse,
                                    opt(Separator::parse),
                                ),
                                cchar(')'),
                            ),
                        )),
                        |(_, _, i)| LineFilterType::Ip(i),
                    ),
                )),
            )),
            |(filter, _, filter_type)| match filter_type {
                LineFilterType::Normal(i) => Self {
                    filter,
                    filter_ip: false,
                    val: i,
                },
                LineFilterType::Ip(i) => Self {
                    filter,
                    filter_ip: true,
                    val: i,
                },
            },
        )(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct LabelParserExpr(LabelParserType);

impl Parser for LabelParserExpr {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("LabelParserExpr {}", input);
        map(LabelParserType::parse, Self)(input)
    }
}

// LabelParserType      ::= 'json' | 'logfmt' | 'regexp Value' | 'unpack' | 'pattern Value'
#[derive(Eq, PartialEq, Debug, Clone)]
pub enum LabelParserType {
    JSON,
    LOGFMT,
    REGEXP(Value),
    UNPACK,
    PATTERN(Value),
}

impl Parser for LabelParserType {
    fn parse(input: &str) -> IResult<&str, Self> {
        alt((
            map(tag("json"), |_| LabelParserType::JSON),
            map(tag("logfmt"), |_| LabelParserType::LOGFMT),
            map(tag("unpack"), |_| LabelParserType::UNPACK),
            map(
                tuple((tag("regexp"), Separator::parse, Value::parse)),
                |(_, _, value)| LabelParserType::REGEXP(value),
            ),
            map(
                tuple((tag("pattern"), Separator::parse, Value::parse)),
                |(_, _, value)| LabelParserType::PATTERN(value),
            ),
        ))(input)
    }
}

#[derive(From, Eq, PartialEq, Debug, Clone)]
pub struct Matchers(Vec<Matcher>);

impl Parser for Matchers {
    fn parse(input: &str) -> IResult<&str, Self> {
        let (remains, matchers) = separated_list1(
            tag(","),
            delimited(opt(Separator::parse), Matcher::parse, opt(Separator::parse)),
        )(input)?;
        Ok((remains, Self(matchers)))
    }
}

// Matcher      ::= Identifier MatchOp Value
#[derive(Eq, PartialEq, Debug, Clone)]
pub struct Matcher {
    pub op: MatchOp,
    pub key: Identifier,
    pub val: Value,
}

impl Parser for Matcher {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            tuple((
                Identifier::parse,
                opt(Separator::parse),
                MatchOp::parse,
                opt(Separator::parse),
                Value::parse,
            )),
            |(key, _, op, _, val)| Self { key, op, val },
        )(input)
    }
}

impl From<(Filter, bool, &str)> for LineFilter {
    fn from(from: (Filter, bool, &str)) -> Self {
        Self {
            filter: from.0,
            filter_ip: from.1,
            val: Value::from(from.2.to_string()),
        }
    }
}

impl TryFrom<(&str, &str, crate::MatchOp)> for IpTableFilter {
    type Error = std::net::AddrParseError;

    fn try_from(from: (&str, &str, crate::MatchOp)) -> Result<Self, Self::Error> {
        Ok(Self {
            key: Identifier::from(from.0.to_string()),
            val: Ip::from(from.1.parse::<std::net::IpAddr>()?),
            op: from.2,
        })
    }
}

impl From<(&str, &str, crate::MatchOp)> for Matcher {
    fn from(from: (&str, &str, MatchOp)) -> Self {
        Self {
            key: Identifier::from(from.0.to_string()),
            val: Value::from(from.1.to_string()),
            op: from.2,
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_matcher() {
        assert_eq!(
            Matcher::parse(r#"abc="123""#).unwrap().1,
            Matcher {
                op: MatchOp::EQ,
                key: Identifier::from("abc".to_string()),
                val: Value::from("123".to_string())
            }
        );

        assert_eq!(
            Matcher::parse(r#"abc     =   "123""#).unwrap().1,
            Matcher {
                op: MatchOp::EQ,
                key: Identifier::from("abc".to_string()),
                val: Value::from("123".to_string())
            }
        )
    }

    #[test]
    fn test_matchers() {
        assert_eq!(
            Matchers::parse(r#"abc="123"  ,   abc     =   "123""#)
                .unwrap()
                .1,
            Matchers(vec![
                ("abc", "123", MatchOp::EQ).into(),
                ("abc", "123", MatchOp::EQ).into(),
            ])
        )
    }

    #[test]
    fn test_label_parser_type() {
        assert_eq!(
            LabelParserType::parse("json").unwrap().1,
            LabelParserType::JSON
        );
        assert_eq!(
            LabelParserType::parse("unpack").unwrap().1,
            LabelParserType::UNPACK
        );
        assert_eq!(
            LabelParserType::parse("logfmt").unwrap().1,
            LabelParserType::LOGFMT
        );
        assert_eq!(
            LabelParserType::parse(r#"regexp    "abc""#).unwrap().1,
            LabelParserType::REGEXP(Value::from("abc".to_string()))
        );
        assert_eq!(
            LabelParserType::parse(r#"pattern    "abc""#).unwrap().1,
            LabelParserType::PATTERN(Value::from("abc".to_string()))
        );
    }

    #[test]
    fn test_label_parser_expr() {
        assert_eq!(
            LabelParserExpr::parse("json").unwrap().1,
            LabelParserExpr(LabelParserType::JSON),
        );
        assert_eq!(
            LabelParserExpr::parse("unpack").unwrap().1,
            LabelParserExpr(LabelParserType::UNPACK)
        );
        assert_eq!(
            LabelParserExpr::parse("logfmt").unwrap().1,
            LabelParserExpr(LabelParserType::LOGFMT)
        );
        assert_eq!(
            LabelParserExpr::parse(r#"regexp    "abc""#).unwrap().1,
            LabelParserExpr(LabelParserType::REGEXP(Value::from("abc".to_string())))
        );
        assert_eq!(
            LabelParserExpr::parse(r#"pattern    "abc""#).unwrap().1,
            LabelParserExpr(LabelParserType::PATTERN(Value::from("abc".to_string())))
        );
    }

    #[test]
    fn test_line_filter() {
        assert_eq!(
            LineFilter::parse(r#"|=      "abc""#).unwrap().1,
            (Filter::PipeExact, false, "abc").into()
        );
        assert_eq!(
            LineFilter::parse(r#"!~    ip  (   "abc"    )"#).unwrap().1,
            (Filter::NRE, true, "abc").into(),
        );
    }

    #[test]
    fn test_line_filters() {
        assert_eq!(
            LineFilters::parse(r#"|=      "abc"  !~    ip  (   "abc"    )"#)
                .unwrap()
                .1,
            LineFilters(vec![
                (Filter::PipeExact, false, "abc").into(),
                (Filter::NRE, true, "abc").into(),
            ])
        );
    }

    #[test]
    fn test_filter() {
        assert_eq!(Filter::parse("|~").unwrap().1, Filter::PipeMatch);
        assert_eq!(Filter::parse("|=").unwrap().1, Filter::PipeExact);
        assert_eq!(Filter::parse("!=").unwrap().1, Filter::NEQ);
        assert_eq!(Filter::parse("!~").unwrap().1, Filter::NRE);
    }

    #[test]
    fn test_label_format() {
        assert_eq!(
            LabelFormat::parse(r#"abc  = "123""#).unwrap().1,
            LabelFormat::Template((
                Identifier::from("abc".to_string()),
                Value::from("123".to_string())
            ))
        );
        assert_eq!(
            LabelFormat::parse(r#"abc  = d"#).unwrap().1,
            LabelFormat::Rename((
                Identifier::from("abc".to_string()),
                Identifier::from("d".to_string())
            ))
        );
    }

    #[test]
    fn test_label_formats() {
        assert_eq!(
            LabelsFormat::parse(r#"abc  = "123"   ,   m  = d  , mk="haha""#)
                .unwrap()
                .1,
            LabelsFormat(vec![
                LabelFormat::Template((
                    Identifier::from("abc".to_string()),
                    Value::from("123".to_string())
                )),
                LabelFormat::Rename((
                    Identifier::from("m".to_string()),
                    Identifier::from("d".to_string())
                )),
                LabelFormat::Template((
                    Identifier::from("mk".to_string()),
                    Value::from("haha".to_string())
                ))
            ])
        );
    }

    #[test]
    fn test_line_format_expr() {
        assert_eq!(
            LineFormatExpr::parse(r#"line_format   "123""#).unwrap().1,
            LineFormatExpr(Value::from("123".to_string()))
        );
    }

    #[test]
    fn test_number_filter() {
        assert_eq!(
            NumberFilter::parse(r#"abc ==    1.234"#).unwrap().1,
            NumberFilter {
                op: LabelFilterType::EQ,
                key: Identifier::from("abc".to_string()),
                val: LiteralExpr::from(1.234)
            }
        );
        assert_eq!(
            NumberFilter::parse(r#"abc!=1.234"#).unwrap().1,
            NumberFilter {
                op: LabelFilterType::NEQ,
                key: Identifier::from("abc".to_string()),
                val: LiteralExpr::from(1.234)
            }
        );
    }

    #[test]
    fn test_bytes_filter() {
        assert_eq!(
            BytesFilter::parse(r#"abc ==    1.234B"#).unwrap().1,
            BytesFilter {
                inner: NumberFilter {
                    op: LabelFilterType::EQ,
                    key: Identifier::from("abc".to_string()),
                    val: (LiteralExpr::from(1.234), BytesUnit::B),
                },
            }
        );
    }

    #[test]
    fn test_duration_filter() {
        assert_eq!(
            DurationFilter::parse(r#"abc ==    1.234ns"#).unwrap().1,
            DurationFilter {
                inner: NumberFilter {
                    op: LabelFilterType::EQ,
                    key: Identifier::from("abc".to_string()),
                    val: (LiteralExpr::from(1.234), DurationUnit::NS),
                },
            }
        );
    }

    #[test]
    fn test_unit_filter() {
        assert_eq!(
            UnitFilter::parse(r#"abc ==    1.234ns"#).unwrap().1,
            UnitFilter::Duration(DurationFilter {
                inner: NumberFilter {
                    op: LabelFilterType::EQ,
                    key: Identifier::from("abc".to_string()),
                    val: (LiteralExpr::from(1.234), DurationUnit::NS),
                },
            })
        );

        assert_eq!(
            UnitFilter::parse(r#"abc ==    1.234KB"#).unwrap().1,
            UnitFilter::Bytes(BytesFilter {
                inner: NumberFilter {
                    op: LabelFilterType::EQ,
                    key: Identifier::from("abc".to_string()),
                    val: (LiteralExpr::from(1.234), BytesUnit::KB),
                },
            })
        );
    }

    #[test]
    fn test_ip_table_filter() {
        assert_eq!(
            IpTableFilter::parse(r#"abc =    ip ( "1.1.1.1"   )  "#)
                .unwrap()
                .1,
            IpTableFilter {
                key: Identifier::from("abc".to_string()),
                val: Ip::from("1.1.1.1".parse::<std::net::IpAddr>().unwrap()),
                op: MatchOp::EQ
            }
        );
    }

    #[test]
    fn test_label_filter() {
        assert_eq!(
            LabelFilter::parse(r#"abc ==    1.234ns"#).unwrap().1,
            LabelFilter::Unit(UnitFilter::Duration(DurationFilter {
                inner: NumberFilter {
                    op: LabelFilterType::EQ,
                    key: Identifier::from("abc".to_string()),
                    val: (LiteralExpr::from(1.234), DurationUnit::NS),
                },
            }))
        );
    }

    #[test]
    fn test_json_expression() {
        assert_eq!(
            JsonExpression::parse(r#"abc = "tag""#).unwrap().1,
            JsonExpression((
                Identifier::from("abc".to_string()),
                Value::from("tag".to_string())
            ))
        );
        assert_eq!(
            JsonExpression::parse(r#"m="ac""#).unwrap().1,
            JsonExpression((
                Identifier::from("m".to_string()),
                Value::from("ac".to_string())
            ))
        )
    }

    #[test]
    fn test_json_expression_parser() {
        assert_eq!(
            JsonExpressionParser::parse(r#"json   abc  = "tag",   m="ac"   , d = "haha""#)
                .unwrap()
                .1,
            JsonExpressionParser(JsonExpressionList(vec![
                JsonExpression((
                    Identifier::from("abc".to_string()),
                    Value::from("tag".to_string())
                )),
                JsonExpression((
                    Identifier::from("m".to_string()),
                    Value::from("ac".to_string())
                )),
                JsonExpression((
                    Identifier::from("d".to_string()),
                    Value::from("haha".to_string())
                ))
            ])),
        );
    }

    #[test]
    fn test_pipeline_storage() {
        assert_eq!(
            PipelineStage::parse(r#"| json   abc  = "tag",   m="ac"   , d = "haha""#)
                .unwrap()
                .1,
            PipelineStage::JsonExpressionParser(JsonExpressionParser(JsonExpressionList(vec![
                JsonExpression((
                    Identifier::from("abc".to_string()),
                    Value::from("tag".to_string())
                )),
                JsonExpression((
                    Identifier::from("m".to_string()),
                    Value::from("ac".to_string())
                )),
                JsonExpression((
                    Identifier::from("d".to_string()),
                    Value::from("haha".to_string())
                ))
            ]))),
        );

        assert_eq!(
            PipelineStage::parse(r#"|=      "abc"  !~    ip  (   "abc"    )"#)
                .unwrap()
                .1,
            PipelineStage::LineFilters(LineFilters(vec![
                LineFilter {
                    filter: Filter::PipeExact,
                    filter_ip: false,
                    val: Value::from("abc".to_string())
                },
                LineFilter {
                    filter: Filter::NRE,
                    filter_ip: true,
                    val: Value::from("abc".to_string())
                }
            ]))
        );
    }

    #[test]
    fn test_pipeline_expr() {
        assert_eq!(
            PipelineExpr::parse(
                r#"| json   abc  = "tag",   m="ac"   , d = "haha"    |="abc"!~ip  (   "abc"    )"#
            )
            .unwrap()
            .1,
            PipelineExpr(vec![
                PipelineStage::JsonExpressionParser(JsonExpressionParser(JsonExpressionList(
                    vec![
                        JsonExpression((
                            Identifier::from("abc".to_string()),
                            Value::from("tag".to_string())
                        )),
                        JsonExpression((
                            Identifier::from("m".to_string()),
                            Value::from("ac".to_string())
                        )),
                        JsonExpression((
                            Identifier::from("d".to_string()),
                            Value::from("haha".to_string())
                        ))
                    ]
                ))),
                PipelineStage::LineFilters(LineFilters(vec![
                    LineFilter {
                        filter: Filter::PipeExact,
                        filter_ip: false,
                        val: Value::from("abc".to_string())
                    },
                    LineFilter {
                        filter: Filter::NRE,
                        filter_ip: true,
                        val: Value::from("abc".to_string())
                    }
                ]))
            ])
        )
    }

    #[test]
    fn test_log_expr() {
        assert_eq!(
            LogExpr::parse(r#"{foo="bar"} |= "baz" |~ "blip" != "flip" !~ "flap" | regexp "(?P<method>\\w+) (?P<path>[\\w|/]+) \\((?P<status>\\d+?)\\) (?P<duration>.*)" | b=ip("127.0.0.1") | level="error" | c=ip("::1") and method!~"2..""#,).unwrap().1,
            LogExpr::PipelineExpr(
                Selector(Some(Matchers(vec![Matcher {
                    op: MatchOp::EQ,
                    key: Identifier::from("foo".to_string()),
                    val: Value::from("bar".to_string())
                }]))),
                PipelineExpr(vec![PipelineStage::LineFilters(LineFilters(
                    vec![
                        (Filter::PipeExact, false, "baz").into(),
                        (Filter::PipeMatch, false, "blip").into(),
                        (Filter::NEQ, false, "flip").into(),
                        (Filter::NRE, false, "flap").into(),
                    ],
                )),
                PipelineStage::LabelParser(LabelParserExpr(LabelParserType::REGEXP(
                    Value::from(r#"(?P<method>\\w+) (?P<path>[\\w|/]+) \\((?P<status>\\d+?)\\) (?P<duration>.*)"#.to_string())
                ))),
                PipelineStage::LabelFilters(LabelFilters::Single(LabelFilter::IpTable(("b", "127.0.0.1", MatchOp::EQ).try_into().unwrap()),)),
                PipelineStage::LabelFilters(LabelFilters::Single(LabelFilter::Matcher(("level", "error", MatchOp::EQ).into()),)),
                PipelineStage::LabelFilters(LabelFilters::Compose {
                    op: LabelFilterOp::AND,
                    lhs: Box::new(LabelFilters::Single(LabelFilter::IpTable(("c", "::1", MatchOp::EQ).try_into().unwrap()),)),
                    rhs: Box::new(LabelFilters::Single(LabelFilter::Matcher(("method", "2..", MatchOp::NRE).into()))) }),
                ])
            )
        );

        assert_eq!(
            LogExpr::parse(r#"{app="foo"} |= "bar"  | logfmt| latency >= 250ms or ( status_code < 500 and status_code > 200)"#).unwrap().1,
            LogExpr::PipelineExpr(
                Selector(Some(Matchers(vec![("app", "foo", MatchOp::EQ).into()]))),
                PipelineExpr(vec![
                    PipelineStage::LineFilters(LineFilters(vec![LineFilter{ filter: Filter::PipeExact, filter_ip: false, val: Value::from("bar".to_string()) }])),
                    PipelineStage::LabelParser(LabelParserExpr(LabelParserType::LOGFMT)),
                    PipelineStage::LabelFilters(LabelFilters::Compose{
                        op: LabelFilterOp::OR,
                        lhs: Box::new(LabelFilters::Single(
                            LabelFilter::Unit(
                                UnitFilter::Duration(
                                    DurationFilter{
                                        inner: NumberFilter {
                                            op: LabelFilterType::GTE,
                                            key: Identifier::from("latency".to_string()),
                                            val: (LiteralExpr::from(250.0), DurationUnit::MS)
                                        }
                                    }
                                )
                            )
                        )),
                        rhs: Box::new(LabelFilters::Compose{
                            op: LabelFilterOp::AND,
                            lhs: Box::new(
                                LabelFilters::Single(
                                    LabelFilter::Number(
                                        NumberFilter {
                                            op: LabelFilterType::LT,
                                            key: Identifier::from("status_code".to_string()),
                                            val: LiteralExpr::from(500.0),
                                        }
                                    )
                                )
                            ),
                            rhs: Box::new(
                                LabelFilters::Single(
                                    LabelFilter::Number(
                                        NumberFilter {
                                            op: LabelFilterType::GT,
                                            key: Identifier::from("status_code".to_string()),
                                            val: LiteralExpr::from(200.0),
                                        }
                                    )
                                )
                            ) })
                    })
                ])),
        );

        assert_eq!(
            LogExpr::parse(r#"{name="join", key   != "clickhouse", val=~   "haah"}"#)
                .unwrap()
                .1,
            LogExpr::MatcherExpr(Selector(Some(Matchers(vec![
                ("name", "join", MatchOp::EQ).into(),
                ("key", "clickhouse", MatchOp::NEQ).into(),
                ("val", "haah", MatchOp::RE).into(),
            ]))))
        );

        assert_eq!(
            LogExpr::parse(r#"{name="join", key   != "clickhouse", val=~   "haah"} | json   abc  = "tag",   m="ac"   , d = "haha"    |=      "abc"  !~    ip  (   "abc"    )"#).unwrap().1,
            LogExpr::PipelineExpr(Selector(Some(Matchers(vec![
                ("name", "join", MatchOp::EQ).into(),
                ("key", "clickhouse", MatchOp::NEQ).into(),
                ("val", "haah", MatchOp::RE).into(),
            ]))), PipelineExpr(vec![PipelineStage::JsonExpressionParser(JsonExpressionParser(JsonExpressionList(vec![
                JsonExpression((Identifier::from("abc".to_string()), Value::from("tag".to_string()))),
                JsonExpression((Identifier::from("m".to_string()), Value::from("ac".to_string()))),
                JsonExpression((Identifier::from("d".to_string()), Value::from("haha".to_string())))
            ]))), PipelineStage::LineFilters(LineFilters(vec![
                (Filter::PipeExact, false, "abc").into(),
                (Filter::NRE, true, "abc").into(),
            ]))]))
        );

        assert_eq!(
            LogExpr::parse(r#"{app!="foo",cluster=~".+bar",bar!~".?boo"}"#)
                .unwrap()
                .1,
            LogExpr::MatcherExpr(Selector(Some(Matchers(vec![
                ("app", "foo", MatchOp::NEQ).into(),
                ("cluster", ".+bar", MatchOp::RE).into(),
                ("bar", ".?boo", MatchOp::NRE).into(),
            ])))),
        );

        assert_eq!(
            LogExpr::parse(r#"{app="foo"} |= "foo" !~ "f.*b""#)
                .unwrap()
                .1,
            LogExpr::PipelineExpr(
                Selector(Some(Matchers(vec![("app", "foo", MatchOp::EQ).into(),]))),
                PipelineExpr(vec![PipelineStage::LineFilters(LineFilters(vec![
                    (Filter::PipeExact, false, "foo").into(),
                    (Filter::NRE, false, "f.*b").into(),
                ]))])
            )
        );
    }

    #[test]
    fn test_aa() {
        println!("{:?}", LogExpr::parse(r#"{app="foo"} |= "bar"  | logfmt| latency >= 250ms or ( status_code < 500 and status_code > 200)"#).unwrap().1,)
    }
}
