use std::str::FromStr;

use crate::prelude::*;

#[derive(From, Debug, Clone)]
pub struct LiteralExpr(f64);

impl Parser for LiteralExpr {
    fn parse(input: &str) -> IResult<&str, Self> {
        map_res(
            recognize(tuple((
                opt(alt((cchar('-'), cchar('+')))),
                digit0,
                opt(pair(cchar('.'), digit0)),
            ))),
            |d_str| -> Result<Self, std::num::ParseFloatError> {
                let d = FromStr::from_str(d_str)?;
                Ok(Self(d))
            },
        )(input)
    }
}

impl PartialEq for LiteralExpr {
    fn eq(&self, other: &Self) -> bool {
        float_cmp::approx_eq!(f64, self.0, other.0)
    }
}

// Identifier      ::=  ( Letter | '_' ) ( Letter | Digit | '.' | '_' )*
#[derive(From, Eq, PartialEq, Debug, Clone, Default)]
pub struct Identifier(String);

impl Parser for Identifier {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            recognize(tuple((
                opt(cchar('_')),
                satisfy(|c| c.is_ascii_alphabetic()),
                take_while(|c: char| c.is_ascii_alphanumeric() || c == '.' || c == '_'),
            ))),
            |s: &str| Self(s.to_owned()),
        )(input)
    }
}

// Literal         ::=  ('"' [^"]* '"') | ("'" [^']* "'")
#[derive(From, Eq, PartialEq, Debug, Clone)]
pub struct Value(String);

impl Parser for Value {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            alt((
                delimited(cchar('"'), take_until("\""), cchar('"')),
                delimited(cchar('\''), take_until("'"), cchar('\'')),
            )),
            |s: &str| Self(s.to_owned()),
        )(input)
    }
}

#[derive(From, Eq, PartialEq, Debug, Clone)]
pub struct Ip(std::net::IpAddr);

impl Parser for Ip {
    fn parse(input: &str) -> IResult<&str, Self> {
        map_res(
            delimited(cchar('"'), take_until("\""), cchar('"')),
            |s: &str| -> Result<Self, std::net::AddrParseError> {
                let ip = s.parse()?;
                Ok(Self(ip))
            },
        )(input)
    }
}

// Comment: '#'
#[derive(Eq, PartialEq, Debug, Clone)]
pub struct Comment(String);

impl Parser for Comment {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(preceded(cchar('#'), take_till(|c| c == '\n')), |s: &str| {
            Self(s.to_owned())
        })(input)
    }
}

// Whitespace, Newline, Tab, Comment
#[derive(Eq, PartialEq, Debug, Copy, Clone)]
pub struct Separator;

impl Parser for Separator {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("Separator {}", input);
        map(
            many1(alt((map(multispace1, |_| ()), map(Comment::parse, |_| ())))),
            |_| Self,
        )(input)
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_literal_expr() {
        assert!(LiteralExpr::parse("1.234").unwrap().1 .0 - 1.234 < 1e-8);
        assert_eq!(LiteralExpr::parse("1.234").unwrap().1, LiteralExpr(1.234));
    }

    #[test]
    fn test_identifier() {
        assert_eq!(
            Identifier::parse("_abc1").unwrap().1,
            Identifier("_abc1".to_string())
        );
        assert_eq!(
            Identifier::parse("_ab.c1").unwrap().1,
            Identifier("_ab.c1".to_string())
        );
        assert_eq!(
            Identifier::parse("ab.c1").unwrap().1,
            Identifier("ab.c1".to_string())
        );
        assert!(Identifier::parse("1").is_err());
    }

    #[test]
    fn test_value() {
        assert_eq!(
            Value::parse(r#""abc""#).unwrap().1,
            Value("abc".to_string())
        );
        assert_eq!(
            Value::parse(r#"'abc'"#).unwrap().1,
            Value("abc".to_string())
        );
    }

    #[test]
    fn test_separator() {
        assert_eq!(Separator::parse("      \n     ").unwrap().1, Separator);
        assert_eq!(Separator::parse("   #abac").unwrap().1, Separator);
        assert_eq!(
            Comment::parse("#abc").unwrap().1,
            Comment("abc".to_string())
        );
        let x: IResult<&str, _> = opt(tag("11"))("");
        println!("aaaaa {:?}", x);
    }

    #[test]
    fn test_ip() {
        assert_eq!(
            Ip::parse(r#""127.0.0.1""#).unwrap().1,
            Ip("127.0.0.1".parse().unwrap())
        );

        assert_eq!(
            Ip::parse(r#""::1""#).unwrap().1,
            Ip("::1".parse().unwrap())
        );
    }
}
