use nom::combinator::cond;

use crate::prelude::*;
use crate::ConvOp;
use crate::DurationUnit;
use crate::Identifier;
use crate::LabelFilters;
use crate::LiteralExpr;
use crate::LogExpr;
use crate::PipelineExpr;
use crate::Selector;
use crate::Separator;
use crate::Value;

#[derive(Debug, Clone, PartialEq)]
pub enum MetricExpr {
    Literal(LiteralExpr),
    RangeAggregation(RangeAggregationExpr),
    VectorAggregation(VectorAggregationExpr),
    LabelReplace(LabelReplaceExpr),
}

impl Parser for MetricExpr {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("MetricExpr {}", input);
        alt((
            map(LiteralExpr::parse, MetricExpr::Literal),
            map(RangeAggregationExpr::parse, MetricExpr::RangeAggregation),
            delimited(cchar('('), MetricExpr::parse, cchar(')')),
            map(VectorAggregationExpr::parse, MetricExpr::VectorAggregation),
            map(LabelReplaceExpr::parse, MetricExpr::LabelReplace),
        ))(input)
    }
}

#[derive(PartialEq, Debug, Clone)]
pub struct RangeAggregationExpr {
    pub op: crate::RangeOp,
    pub log_range_expr: LogRangeExpr,
    pub param: Option<LiteralExpr>,
    pub grouping: Option<Grouping>,
}

impl Parser for RangeAggregationExpr {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("RangeAggregationExpr {}", input);
        map(
            tuple((
                crate::RangeOp::parse,
                opt(Separator::parse),
                cchar('('),
                opt(Separator::parse),
                opt(map(
                    tuple((LiteralExpr::parse, opt(Separator::parse), tag(","))),
                    |(param, _, _)| param,
                )),
                opt(Separator::parse),
                LogRangeExpr::parse,
                cchar(')'),
                opt(Separator::parse),
                opt(Grouping::parse),
            )),
            |(op, _, _, _, param, _, log_range_expr, _, _, grouping)| Self {
                op,
                log_range_expr,
                param,
                grouping,
            },
        )(input)
    }
}

#[derive(PartialEq, Debug, Clone)]
pub struct LogRangeExpr {
    pub log_expr: LogExpr,
    pub offset_expr: Option<OffsetExpr>,
    pub unwrap_expr: Option<UnwrapExpr>,
    pub range: Duration,
}

pub struct SelectorPipelineUnwrap {
    selector: Selector,
    pipeline_expr: Option<PipelineExpr>,
    unwrap_expr: Option<UnwrapExpr>,
    range: Duration,
}

impl Parser for SelectorPipelineUnwrap {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("SelectorPipelineUnwrap {}", input);
        map(
            tuple((
                Selector::parse,
                opt(Separator::parse),
                opt(PipelineExpr::parse),
                opt(Separator::parse),
                opt(UnwrapExpr::parse),
                opt(Separator::parse),
                cchar('['),
                opt(Separator::parse),
                Duration::parse,
                opt(Separator::parse),
                cchar(']'),
            )),
            |(selector, _, pipeline_expr, _, unwrap_expr, _, _, _, range, _, _)| Self {
                selector,
                pipeline_expr,
                unwrap_expr,
                range,
            },
        )(input)
    }
}

// 有没有函数式的写法？？这样写很不舒服。。。
impl LogRangeExpr {
    pub fn parse_inner(input: &str) -> IResult<&str, Self> {
        println!("LogRangeExpr {}", input);
        let (input, i) = map(
            alt((
                SelectorPipelineUnwrap::parse,
                map(
                    tuple((cchar('('), SelectorPipelineUnwrap::parse, cchar(')'))),
                    |(_, i, _)| i,
                ),
            )),
            |i| i,
        )(input)?;

        opt(Separator::parse)(input)?;
        let (input, offset_expr) = opt(OffsetExpr::parse)(input)?;
        let (input, pipeline_expr) =
            cond(i.pipeline_expr.is_none(), opt(PipelineExpr::parse))(input)?;
        let (input, unwrap_expr) = cond(i.unwrap_expr.is_none(), opt(UnwrapExpr::parse))(input)?;
        let pipeline_expr = pipeline_expr.unwrap_or(i.pipeline_expr);
        let unwrap_expr = unwrap_expr.unwrap_or(i.unwrap_expr);
        let log_expr = if let Some(pipeline_expr) = pipeline_expr {
            LogExpr::PipelineExpr(i.selector, pipeline_expr)
        } else {
            LogExpr::MatcherExpr(i.selector)
        };
        Ok((
            input,
            Self {
                log_expr,
                offset_expr,
                unwrap_expr,
                range: i.range,
            },
        ))
    }
}

impl Parser for LogRangeExpr {
    fn parse(input: &str) -> IResult<&str, Self> {
        alt((
            LogRangeExpr::parse_inner,
            map(
                tuple((
                    cchar('('),
                    opt(Separator::parse),
                    Self::parse,
                    opt(Separator::parse),
                    cchar(')'),
                )),
                |(_, _, expr, _, _)| expr,
            ),
        ))(input)
    }
}
#[derive(PartialEq, Debug, Clone)]
pub struct OffsetExpr(Duration);

impl Parser for OffsetExpr {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            tuple((tag("offset"), LiteralExpr::parse, DurationUnit::parse)),
            |(_, i, j)| Self(Duration(i, j)),
        )(input)
    }
}

#[derive(PartialEq, Debug, Clone)]
pub struct Duration(pub LiteralExpr, pub DurationUnit);

impl Parser for Duration {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            tuple((LiteralExpr::parse, DurationUnit::parse)),
            |(i, j)| Self(i, j),
        )(input)
    }
}
#[derive(PartialEq, Debug, Clone)]
pub struct UnwrapExpr {
    id: Identifier,
    operation: Option<ConvOp>,
    post_filters: Option<LabelFilters>,
}

impl Parser for UnwrapExpr {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("UnwrapExpr {}", input);
        map(
            tuple((
                map(
                    tuple((
                        tag("|"),
                        opt(Separator::parse),
                        tag("unwrap"),
                        opt(Separator::parse),
                        alt((
                            map(Identifier::parse, |id| Self {
                                id,
                                operation: None,
                                post_filters: None,
                            }),
                            map(
                                tuple((
                                    ConvOp::parse,
                                    opt(Separator::parse),
                                    cchar('('),
                                    opt(Separator::parse),
                                    Identifier::parse,
                                    opt(Separator::parse),
                                    cchar(')'),
                                )),
                                |(op, _, _, _, id, _, _)| Self {
                                    id,
                                    operation: Some(op),
                                    post_filters: None,
                                },
                            ),
                        )),
                    )),
                    |(_, _, _, _, res)| res,
                ),
                opt(Separator::parse),
                opt(map(
                    tuple((cchar('|'), opt(Separator::parse), LabelFilters::parse)),
                    |(_, _, label_filter)| label_filter,
                )),
            )),
            |(i, _, label_filter)| Self {
                id: i.id,
                operation: i.operation,
                post_filters: label_filter,
            },
        )(input)
    }
}

#[derive(derive_newtype::NewType, Eq, PartialEq, Debug, Clone)]
pub struct Labels(Vec<crate::Identifier>);

impl Parser for Labels {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("Labels {}", input);
        let (remains, labels) = separated_list1(
            tag(","),
            map(
                tuple((
                    opt(Separator::parse),
                    crate::Identifier::parse,
                    opt(Separator::parse),
                )),
                |(_, label, _)| label,
            ),
        )(input)?;
        Ok((remains, Self(labels)))
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct Grouping {
    pub without: bool,
    pub groups: Option<Labels>,
}

impl Parser for Grouping {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("Grouping {}", input);
        map(
            tuple((
                alt((map(tag("by"), |_| false), map(tag("without"), |_| true))),
                opt(Separator::parse),
                cchar('('),
                opt(Labels::parse),
                opt(Separator::parse),
                cchar(')'),
            )),
            |(without, _, _, groups, _, _)| Self { without, groups },
        )(input)
    }
}

#[derive(PartialEq, Debug, Clone)]
pub struct VectorAggregationExpr {
    pub left: Box<MetricExpr>,
    pub op: crate::VectorOp,
    pub grouping: Option<Grouping>,
    pub param: Option<LiteralExpr>,
}

impl Parser for VectorAggregationExpr {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("VectorAggregationExpr {}", input);
        map(
            tuple((
                crate::VectorOp::parse,
                opt(Separator::parse),
                alt((
                    map(
                        tuple((
                            Grouping::parse,
                            opt(Separator::parse),
                            cchar('('),
                            opt(Separator::parse),
                            opt(map(
                                tuple((LiteralExpr::parse, opt(Separator::parse), tag(","))),
                                |(param, _, _)| param,
                            )),
                            opt(Separator::parse),
                            MetricExpr::parse,
                            opt(Separator::parse),
                            cchar(')'),
                        )),
                        |(grouping, _, _, _, param, _, left, _, _)| (Some(grouping), param, left),
                    ),
                    map(
                        tuple((
                            cchar('('),
                            opt(Separator::parse),
                            opt(map(
                                tuple((LiteralExpr::parse, opt(Separator::parse), tag(","))),
                                |(param, _, _)| param,
                            )),
                            opt(Separator::parse),
                            MetricExpr::parse,
                            opt(Separator::parse),
                            cchar(')'),
                            opt(Separator::parse),
                            opt(Grouping::parse),
                        )),
                        |(_, _, param, _, left, _, _, _, grouping)| (grouping, param, left),
                    ),
                )),
            )),
            |(op, _, i)| Self {
                left: Box::new(i.2),
                op,
                grouping: i.0,
                param: i.1,
            },
        )(input)
    }
}

#[derive(PartialEq, Debug, Clone)]
pub struct LabelReplaceExpr {
    left: Box<MetricExpr>,
    dst: Value,
    replacement: Value,
    src: Value,
    regex: Value,
}

impl Parser for LabelReplaceExpr {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            tuple((
                tag("label_replace"),
                opt(Separator::parse),
                cchar('('),
                opt(Separator::parse),
                MetricExpr::parse,
                opt(Separator::parse),
                tag(","),
                opt(Separator::parse),
                Value::parse,
                opt(Separator::parse),
                tag(","),
                Value::parse,
                opt(Separator::parse),
                tag(","),
                Value::parse,
                opt(Separator::parse),
                tag(","),
                Value::parse,
                opt(Separator::parse),
                cchar(')'),
            )),
            |(_, _, _, _, left, _, _, _, dst, _, _, replacement, _, _, src, _, _, regex, _, _)| {
                Self {
                    left: Box::new(left),
                    dst,
                    replacement,
                    src,
                    regex,
                }
            },
        )(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct BoolModifier(pub(crate) CardType, pub(crate) bool);

impl Parser for BoolModifier {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(opt(tag("bool")), |return_bool| {
            Self(CardType::OneToOne, return_bool.is_some())
        })(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct OnOrIgnoringModifier {
    pub(crate) return_bool: BoolModifier,
    pub(crate) on: bool,
    pub(crate) labels: Option<Labels>,
}

impl Parser for OnOrIgnoringModifier {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            tuple((
                BoolModifier::parse,
                opt(Separator::parse),
                alt((map(tag("on"), |_| true), map(tag("ignoring"), |_| false))),
                opt(Separator::parse),
                cchar('('),
                opt(Separator::parse),
                opt(Labels::parse),
                opt(Separator::parse),
                cchar(')'),
            )),
            |(return_bool, _, on, _, _, _, labels, _, _)| Self {
                return_bool,
                on,
                labels,
            },
        )(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub enum CardType {
    OneToOne,
    ManyToOne,
    OneToMany,
}
#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_literal_expr() {
        assert_eq!(
            MetricExpr::parse("1.234").unwrap().1,
            MetricExpr::Literal(LiteralExpr::from(1.234))
        );
        assert_eq!(
            MetricExpr::parse("(((((1.234)))))").unwrap().1,
            MetricExpr::Literal(LiteralExpr::from(1.234))
        );
    }
}
