use crate::prelude::*;
use crate::BinOp;
use crate::BoolModifier;
use crate::CardType;
use crate::Labels;
use crate::LogExpr;
use crate::MetricExpr;
use crate::OnOrIgnoringModifier;
use crate::Separator;

#[derive(Debug, Clone, PartialEq)]
pub enum Expr {
    Metric(MetricExpr),
    Log(LogExpr),
    BinOp {
        op: BinOp,
        options: BinOpOptions,
        lhs: Box<Expr>,
        rhs: Box<Expr>,
    },
}

impl Parser for Expr {
    fn parse(input: &str) -> IResult<&str, Self> {
        map(
            tuple((
                alt((
                    map(MetricExpr::parse, Expr::Metric),
                    map(LogExpr::parse, Expr::Log),
                )),
                opt(map(
                    tuple((
                        opt(Separator::parse),
                        BinOp::parse,
                        opt(Separator::parse),
                        BinOpOptions::parse,
                        opt(Separator::parse),
                        Expr::parse,
                    )),
                    |(_, op, _, options, _, expr)| (op, options, expr),
                )),
            )),
            |(lhs, right)| match right {
                Some((op, options, rhs)) => Self::BinOp {
                    op,
                    options,
                    lhs: Box::new(lhs),
                    rhs: Box::new(rhs),
                },
                None => lhs,
            },
        )(input)
    }
}

#[derive(Eq, PartialEq, Debug, Clone)]
pub struct BinOpOptions {
    return_bool: bool,
    card: CardType,
    matching_labels: Option<Labels>,
    on: bool,
    include: Option<Labels>,
}

impl Parser for BinOpOptions {
    fn parse(input: &str) -> IResult<&str, Self> {
        println!("BinOpOptions {}", input);
        alt((
            map(
                tuple((
                    OnOrIgnoringModifier::parse,
                    opt(Separator::parse),
                    opt(alt((
                        map(tag("group_left"), |_| CardType::ManyToOne),
                        map(tag("group_right"), |_| CardType::OneToMany),
                    ))),
                    opt(Separator::parse),
                    opt(map(
                        tuple((
                            cchar('('),
                            opt(Separator::parse),
                            opt(Labels::parse),
                            opt(Separator::parse),
                            cchar(')'),
                        )),
                        |(_, _, labels, _, _)| labels,
                    )),
                )),
                |(on_or_ignoring, _, card_type, _, include)| {
                    let card_type = match card_type {
                        Some(card_type) => card_type,
                        None => on_or_ignoring.return_bool.0,
                    };
                    let include = match include {
                        Some(include) => include,
                        None => None,
                    };
                    Self {
                        return_bool: on_or_ignoring.return_bool.1,
                        card: card_type,
                        matching_labels: on_or_ignoring.labels,
                        on: on_or_ignoring.on,
                        include,
                    }
                },
            ),
            map(BoolModifier::parse, |i| Self {
                return_bool: i.1,
                card: i.0,
                matching_labels: None,
                on: false,
                include: None,
            }),
        ))(input)
    }
}

#[cfg(test)]
mod tests {
    use crate::{
        basic::LiteralExpr, Duration, DurationUnit, Grouping, Identifier, LogRangeExpr, MatchOp,
        Matcher, Matchers, RangeAggregationExpr, RangeOp, Selector, Value, VectorAggregationExpr,
        VectorOp,
    };

    use super::*;

    #[test]
    fn test_expr() {
        assert_eq!(
            Expr::parse("1.234").unwrap().1,
            Expr::Metric(MetricExpr::Literal(LiteralExpr::from(1.234)))
        );

        assert_eq!(
            Expr::parse("(((1.234)))").unwrap().1,
            Expr::Metric(MetricExpr::Literal(LiteralExpr::from(1.234)))
        );

        assert_eq!(
            Expr::parse(r#"topk(10, sum(rate({region="us-east1"} [5m]))) by (name)"#)
                .unwrap()
                .1,
            Expr::Metric(MetricExpr::VectorAggregation(VectorAggregationExpr {
                left: Box::new(MetricExpr::VectorAggregation(VectorAggregationExpr {
                    left: Box::new(MetricExpr::RangeAggregation(RangeAggregationExpr {
                        op: RangeOp::Rate,
                        log_range_expr: LogRangeExpr {
                            log_expr: LogExpr::MatcherExpr(Selector::from(Some(Matchers::from(
                                vec![Matcher {
                                    op: MatchOp::EQ,
                                    key: Identifier::from("region".to_string()),
                                    val: Value::from("us-east1".to_string())
                                }]
                            )))),
                            offset_expr: None,
                            unwrap_expr: None,
                            range: Duration(LiteralExpr::from(5.0), DurationUnit::M)
                        },
                        param: None,
                        grouping: None,
                    })),
                    op: VectorOp::SUM,
                    grouping: None,
                    param: None,
                })),
                op: VectorOp::TOPK,
                grouping: Some(Grouping {
                    without: false,
                    groups: Some(Labels::from(vec![Identifier::from("name".to_string())]))
                }),
                param: Some(LiteralExpr::from(10.0))
            }))
        );

        assert_eq!(
            Expr::parse(r#"count_over_time({job="mysql"} [5m])"#)
                .unwrap()
                .1,
            Expr::Metric(MetricExpr::RangeAggregation(RangeAggregationExpr {
                op: RangeOp::Count,
                log_range_expr: LogRangeExpr {
                    log_expr: LogExpr::MatcherExpr(Selector::from(Some(Matchers::from(
                        vec![Matcher {
                            op: MatchOp::EQ,
                            key: Identifier::from("job".to_string()),
                            val: Value::from("mysql".to_string()),
                        }]
                    )))),
                    offset_expr: None,
                    unwrap_expr: None,
                    range: Duration(LiteralExpr::from(5.0), DurationUnit::M),
                },
                param: None,
                grouping: None,
            }))
        );

        println!("{:?}", Expr::parse(r#"json response_status="response.status.code",ip="182.168.0.1""#));
    }

    #[test]
    fn test_bin_op() {
        assert_eq!(
            Expr::parse(r#"sum(count_over_time({app="foo"} [1m])) by (app, machine) > bool on(app) group_left(pool) sum(count_over_time({app="foo"} [1m])) by (app, pool)"#).unwrap().1,
            Expr::BinOp {
                op: BinOp::GT,
                options: BinOpOptions {
                    return_bool: true,
                    card: CardType::ManyToOne,
                    matching_labels: Some(Labels::from(vec![Identifier::from("app".to_string())])),
                    on: true,
                    include: Some(Labels::from(vec![Identifier::from("pool".to_string())])),
                },
                lhs: Box::new(Expr::Metric(MetricExpr::VectorAggregation(
                    VectorAggregationExpr {
                        left: Box::new(MetricExpr::RangeAggregation(RangeAggregationExpr {
                            op: RangeOp::Count,
                            log_range_expr: LogRangeExpr {
                                log_expr: LogExpr::MatcherExpr(Selector::from(Some(
                                    Matchers::from(vec![Matcher {
                                        op: MatchOp::EQ,
                                        key: Identifier::from("app".to_string()),
                                        val: Value::from("foo".to_string()),
                                    }])
                                ))),
                                offset_expr: None,
                                unwrap_expr: None,
                                range: Duration(LiteralExpr::from(1.0), DurationUnit::M)
                            },
                            param: None,
                            grouping: None
                        })),
                        op: VectorOp::SUM,
                        grouping: Some(Grouping {
                            without: false,
                            groups: Some(Labels::from(vec![
                                Identifier::from("app".to_string()),
                                Identifier::from("machine".to_string())
                            ]))
                        }),
                        param: None,
                    }
                ))),
                rhs: Box::new(Expr::Metric(MetricExpr::VectorAggregation(
                    VectorAggregationExpr {
                        left: Box::new(MetricExpr::RangeAggregation(RangeAggregationExpr {
                            op: RangeOp::Count,
                            log_range_expr: LogRangeExpr {
                                log_expr: LogExpr::MatcherExpr(Selector::from(Some(
                                    Matchers::from(vec![Matcher {
                                        op: MatchOp::EQ,
                                        key: Identifier::from("app".to_string()),
                                        val: Value::from("foo".to_string()),
                                    }])
                                ))),
                                offset_expr: None,
                                unwrap_expr: None,
                                range: Duration(LiteralExpr::from(1.0), DurationUnit::M)
                            },
                            param: None,
                            grouping: None
                        })),
                        op: VectorOp::SUM,
                        grouping: Some(Grouping {
                            without: false,
                            groups: Some(Labels::from(vec![
                                Identifier::from("app".to_string()),
                                Identifier::from("pool".to_string())
                            ]))
                        }),
                        param: None,
                    }
                )))
            }
        );
    }
}
