pub mod postgresql {
    use postgres::{Client, Error, NoTls};
    use serde_json::json;
    extern crate redis;

    /// Current connection string for PostgreSQL from Redis
    pub fn db_connect() -> String {
        let client = redis::Client::open("redis://127.0.0.1/").unwrap();
        let mut con = client.get_connection().unwrap();
        let y: String = redis::cmd("GET")
            .arg("db_string_connection")
            .query(&mut con)
            .unwrap();
        return y;
    }

    /// Set default transaction isolation level for database
    pub fn set_transaction_isolation() -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.batch_execute(
            "alter database postgres set default_transaction_isolation to serializable",
        )?;
        Ok(())
    }

    /// Create SQL table "plc" for new PLC
    pub fn create_plc_table(table_name: &str) -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        let mut sql_query = String::from("CREATE TABLE IF NOT EXISTS ");
        sql_query.push_str(table_name);
        sql_query.push_str(" (plc_name text NOT NULL, plc_ip text NOT NULL, const text NOT NULL, mark timestamp default current_timestamp)");
        client.batch_execute(&sql_query)?;
        Ok(())
    }

    /// Insert default values in new table "plc"
    pub fn insert_plc_table(table_name: &str, plc_name: &str, plc_ip: &str) -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        let mut sql_query = String::from("INSERT INTO ");
        sql_query.push_str(table_name);
        sql_query.push_str(" (plc_name, plc_ip, const) VALUES ('");
        sql_query.push_str(plc_name);
        sql_query.push_str("', '");
        sql_query.push_str(plc_ip);
        sql_query.push_str("', 'static')");
        client.batch_execute(&sql_query)?;
        Ok(())
    }

    /// Create SQL table "plc_list" for all PLC
    pub fn create_plc_list_table() -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.batch_execute(
            "
                CREATE TABLE IF NOT EXISTS plc_list (
                    new_plc_name text NOT NULL,
                    mark timestamp default current_timestamp
                )
            ",
        )?;
        Ok(())
    }

    /// Create SQL table "var_list" for all PLC
    pub fn create_var_list_table() -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.batch_execute(
            "
                CREATE TABLE IF NOT EXISTS var_list (
                    variable text NOT NULL,
                    mark timestamp default current_timestamp
                )
            ",
        )?;
        Ok(())
    }

    /// Insert new plc name to plc list
    pub fn insert_new_plc_to_plc_list(new_plc_name: &str) -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.execute(
            "INSERT INTO plc_list (new_plc_name) VALUES ($1)",
            &[&new_plc_name],
        )?;
        Ok(())
    }

    pub fn insert_json_variable_to_var_list(plc_path: &str, var_path: &str) -> Result<(), Error> {
        let new_variable = json!({
            "variable_path": [
                plc_path,
                var_path
            ]
        });
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.execute(
            "INSERT INTO var_list (variable) VALUES ($1)",
            &[&new_variable.to_string()],
        )?;
        Ok(())
    }

    /// Create SQL table "variables" for all PLC
    pub fn create_variables_table() -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.batch_execute(
            "
                CREATE TABLE IF NOT EXISTS variables (
                    variable text NOT NULL,
                    mark timestamp default current_timestamp
                )
            ",
        )?;
        Ok(())
    }

    /// Convert JSON new variable to String and insert to SQL table "variables"
    pub fn insert_json_new_variable(
        plc_path: &str,
        var_path: &str,
        addr_dec: &str,
        num_func: &str,
    ) -> Result<(), Error> {
        // The type of `john` is `serde_json::Value`
        let new_variable = json!({
            "variable_path": [
                plc_path,
                var_path
            ],
            "variable_param": [
                addr_dec,
                num_func
            ]
        });
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.execute(
            "INSERT INTO variables (variable) VALUES ($1)",
            &[&new_variable.to_string()],
        )?;
        Ok(())
    }

    pub fn delete_plc_table(table_name: &str) -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        let mut sql_query = String::from("DROP TABLE ");
        sql_query.push_str(table_name);
        client.batch_execute(&sql_query)?;
        Ok(())
    }

    pub fn delete_plc_from_plc_list(plc_name: &str) -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.execute(
            "DELETE FROM plc_list WHERE new_plc_name = ($1)",
            &[&plc_name],
        )?;
        Ok(())
    }

    pub fn delete_plc_from_var_list(plc_name: &str) -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.execute(
            "DELETE FROM var_list WHERE variable::json #>> '{variable_path, 0}' = ($1)",
            &[&plc_name],
        )?;
        Ok(())
    }

    pub fn delete_plc_from_variables(plc_name: &str) -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.execute(
            "DELETE FROM variables WHERE variable::json #>> '{variable_path, 0}' = ($1)",
            &[&plc_name],
        )?;
        Ok(())
    }

    pub fn delete_variable_from_var_list(var_name: &str) -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.execute(
            "DELETE FROM var_list WHERE variable::json #>> '{variable_path, 1}' = ($1)",
            &[&var_name],
        )?;
        Ok(())
    }

    pub fn delete_variable_from_variables(var_name: &str) -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.execute(
            "DELETE FROM variables WHERE variable::json #>> '{variable_path, 1}' = ($1)",
            &[&var_name],
        )?;
        Ok(())
    }

    pub fn new_project(new_project_name: &str) -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        let mut sql_query = String::from("CREATE DATABASE ");
        sql_query.push_str(new_project_name);
        client.batch_execute(&sql_query)?;
        Ok(())
    }

    pub fn create_themes_table() -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.batch_execute(
            "
                CREATE TABLE IF NOT EXISTS themes (
                    theme text NOT NULL,
                    mark timestamp default current_timestamp
                )
            ",
        )?;
        Ok(())
    }

    pub fn insert_theme_to_themes(theme: &str) -> Result<(), Error> {
        let mut client = Client::connect(&db_connect(), NoTls)?;
        client.execute("INSERT INTO themes (theme) VALUES ($1)", &[&theme])?;
        Ok(())
    }
}
