#![windows_subsystem = "windows"]
use fltk::{
    app, dialog,
    enums::{Event, FrameType, Shortcut},
    menu,
    prelude::*,
    tree::{Tree, TreeSelect, TreeSort},
    window,
};
use fltk_theme::{ThemeType, WidgetTheme};
use postgres::{Client, NoTls};

use serde_json::Value;

mod content;
mod my_redis;
mod psql;
mod theme;

#[derive(Copy, Clone)]
pub enum Message {
    NewProject,
    OpenProject,
    Save,
    SaveAs,
    Quit,
    AddPlC,
    DeletePlc,
    AddVariable,
    DeleteVariable,
    GreyBird,
    Metro,
    Dark,
    Run,
    Stop,
}

pub fn center() -> (i32, i32) {
    (
        (app::screen_size().0 / 2.0) as i32,
        (app::screen_size().1 / 2.0) as i32,
    )
}

pub struct MyMenu {
    menu: menu::SysMenuBar,
}

impl MyMenu {
    pub fn new(s: &app::Sender<Message>) -> Self {
        let mut menu = menu::SysMenuBar::default().with_size(800, 35);
        menu.set_frame(FrameType::FlatBox);
        menu.add_emit(
            "&File/New project\t",
            Shortcut::Ctrl | 'n',
            menu::MenuFlag::Normal,
            *s,
            Message::NewProject,
        );

        menu.add_emit(
            "&File/Open project\t",
            Shortcut::Ctrl | 'o',
            menu::MenuFlag::Normal,
            *s,
            Message::OpenProject,
        );

        menu.add_emit(
            "&File/Save\t",
            Shortcut::Ctrl | 's',
            menu::MenuFlag::Normal,
            *s,
            Message::Save,
        );

        menu.add_emit(
            "&File/Save as\t",
            Shortcut::Ctrl | 'w',
            menu::MenuFlag::Normal,
            *s,
            Message::SaveAs,
        );

        menu.add_emit(
            "&File/Quit\t",
            Shortcut::Ctrl | 'q',
            menu::MenuFlag::Normal,
            *s,
            Message::Quit,
        );

        menu.add_emit(
            "&Edit/Add PLC\t",
            Shortcut::Ctrl | 'p',
            menu::MenuFlag::Normal,
            *s,
            Message::AddPlC,
        );

        menu.add_emit(
            "&Edit/Delete PLC\t",
            Shortcut::Ctrl | 'd',
            menu::MenuFlag::Normal,
            *s,
            Message::DeletePlc,
        );

        menu.add_emit(
            "&Edit/Add variable\t",
            Shortcut::Ctrl | 'i',
            menu::MenuFlag::Normal,
            *s,
            Message::AddVariable,
        );

        menu.add_emit(
            "&Edit/Delete variable\t",
            Shortcut::Ctrl | 'u',
            menu::MenuFlag::Normal,
            *s,
            Message::DeleteVariable,
        );

        menu.add_emit(
            "&Theme/Grey Bird\t",
            Shortcut::Ctrl | 'l',
            menu::MenuFlag::Normal,
            *s,
            Message::GreyBird,
        );

        menu.add_emit(
            "&Theme/Metro\t",
            Shortcut::Ctrl | 'l',
            menu::MenuFlag::Normal,
            *s,
            Message::Metro,
        );

        menu.add_emit(
            "&Theme/Dark\t",
            Shortcut::Ctrl | 'l',
            menu::MenuFlag::Normal,
            *s,
            Message::Dark,
        );

        menu.add_emit(
            "&Work/Run\t",
            Shortcut::Ctrl | 'r',
            menu::MenuFlag::Normal,
            *s,
            Message::Run,
        );

        menu.add_emit(
            "&Work/Stop\t",
            Shortcut::Ctrl | 't',
            menu::MenuFlag::Normal,
            *s,
            Message::Stop,
        );

        Self { menu: menu }
    }
}

pub struct MyApp {
    app: app::App,
    modified: bool,
    r: app::Receiver<Message>,
    main_win: window::Window,
    menu: MyMenu,
    tree: Tree,
}

impl MyApp {
    pub fn new(_args: Vec<String>) -> Self {
        let app = app::App::default().with_scheme(app::Scheme::Gtk);
        let widget_theme = WidgetTheme::new(ThemeType::Greybird);
        widget_theme.apply();
        theme::update::theme_type();
        dialog::message_title_default("Logical Systems TCP Modbus Server");
        let (s, r) = app::channel::<Message>();
        let mut main_win = window::Window::default()
            .with_size(800, 600)
            .center_screen()
            .with_label("Logical Systems TCP Modbus Server");
        let menu = MyMenu::new(&s);
        let modified = false;
        let mut tree = Tree::default().with_size(794, 540).center_of_parent();
        tree.set_root_label("PLC list");
        tree.set_select_mode(TreeSelect::Multi);
        tree.set_sort_order(TreeSort::Ascending);
        tree.set_show_collapse(true);
        tree.set_show_root(false);
        main_win.make_resizable(true);
        main_win.end();
        main_win.show();

        let db_host_name_input = dialog::input_default(
            "Greetings!\nThis app uses DBMS Postgresql to store data\nPlease enter the parameters to establish a connection to the DBMS\n\nEnter host name:",
            "localhost",
        );
        if let Some(db_host_name_input) = db_host_name_input {
            let db_name_input = dialog::input_default("Enter maintenance database:", "postgres");
            if let Some(db_name_input) = db_name_input {
                let db_user_name_input = dialog::input_default("Enter username:", "admin");
                if let Some(db_user_name_input) = db_user_name_input {
                    let db_password_input = dialog::input_default("Enter password:", "postgres");
                    if let Some(db_password_input) = db_password_input {
                        my_redis::memory::db_test(
                            &db_name_input,
                            &db_host_name_input,
                            &db_name_input,
                            &db_user_name_input,
                            &db_password_input,
                        );
                        my_redis::memory::db_test_connect(
                            &db_host_name_input,
                            &db_name_input,
                            &db_user_name_input,
                            &db_password_input,
                        );
                        my_redis::memory::write_projects_list(&db_name_input);
                    }
                }
            }
        }

        psql::postgresql::set_transaction_isolation();
        psql::postgresql::create_plc_list_table();
        psql::postgresql::create_var_list_table();
        psql::postgresql::create_variables_table();

        add_tree!(tree);

        main_win.set_callback(move |_| {
            if app::event() == Event::Close {
                s.send(Message::Quit);
            }
        });

        tree.set_callback(|t| {
            if let Some(items) = t.get_selected_items() {
                for i in items {
                    println!("{} selected", t.item_pathname(&i).unwrap());
                }
            }
        });

        Self {
            app,
            modified,
            r,
            main_win,
            menu,
            tree,
        }
    }

    pub fn launch(&mut self) {
        while self.app.wait() {
            use Message::*;
            if let Some(msg) = self.r.recv() {
                match msg {
                    NewProject => {
                        println!("New project");
                        let new_project_name_input =
                            dialog::input_default("Enter project name:", "factory");
                        if let Some(new_project_name_input) = new_project_name_input {
                            let db_host_name_input = dialog::input_default(
                                "Greetings!\nThis app uses DBMS Postgresql to store data\nPlease enter the parameters to establish a connection to the DBMS\n\nEnter host name:",
                                "localhost",
                            );
                            if let Some(db_host_name_input) = db_host_name_input {
                                let db_name_input = dialog::input_default("Your maintenance database = your 'project name'\nPress 'Ok' to continue", &new_project_name_input);
                                if let Some(_db_name_input) = db_name_input {
                                    let db_user_name_input =
                                        dialog::input_default("Enter username:", "admin");
                                    if let Some(db_user_name_input) = db_user_name_input {
                                        let db_password_input =
                                            dialog::input_default("Enter password:", "postgres");
                                        if let Some(db_password_input) = db_password_input {
                                            psql::postgresql::new_project(&new_project_name_input);
                                            my_redis::memory::db_test(
                                                &new_project_name_input,
                                                &db_host_name_input,
                                                &new_project_name_input,
                                                &db_user_name_input,
                                                &db_password_input,
                                            );
                                            my_redis::memory::db_test_connect(
                                                &db_host_name_input,
                                                &new_project_name_input,
                                                &db_user_name_input,
                                                &db_password_input,
                                            );
                                            my_redis::memory::write_projects_list(
                                                &new_project_name_input,
                                            );
                                            self.tree.clear();
                                            psql::postgresql::set_transaction_isolation();
                                            psql::postgresql::create_plc_list_table();
                                            psql::postgresql::create_var_list_table();
                                            psql::postgresql::create_variables_table();
                                            add_tree!(self.tree);
                                        }
                                    }
                                }
                            }
                        }
                    }
                    OpenProject => {
                        println!("Open project");
                        let mut main_win = window::Window::default()
                            .with_size(800, 600)
                            .center_screen()
                            .with_label("Logical Systems TCP Modbus Server");
                        let _modified = false;
                        let mut tree = Tree::default().with_size(794, 540).center_of_parent();
                        tree.set_root_label("PLC list");
                        tree.set_select_mode(TreeSelect::Multi);
                        tree.set_sort_order(TreeSort::Ascending);
                        tree.set_show_collapse(true);
                        tree.set_show_root(false);
                        main_win.make_resizable(true);
                        main_win.end();
                        main_win.show();
                        add_projects_list!(tree);
                    }
                    Save => {
                        println!("Save");
                    }
                    SaveAs => {
                        println!("Save as");
                    }
                    Quit => {
                        if self.modified {
                            match dialog::choice2(
                                center().0 - 200,
                                center().1 - 100,
                                "Would you like to save your work?",
                                "Yes",
                                "No",
                                "",
                            ) {
                                Some(0) => {
                                    self.app.quit();
                                }
                                Some(1) => self.app.quit(),
                                Some(_) | None => (),
                            }
                        } else {
                            self.app.quit();
                        }
                    }
                    AddPlC => {
                        match dialog::choice2_default(
                            "Would you like to add new PLC?",
                            "Yes",
                            "No",
                            "",
                        ) {
                            Some(0) => {
                                let plc_name_input =
                                    dialog::input_default("Enter name PLC:", "Trim5");
                                if let Some(plc_name_input) = plc_name_input {
                                    self.tree.add(&plc_name_input);
                                    psql::postgresql::create_plc_table(&plc_name_input);
                                    psql::postgresql::insert_new_plc_to_plc_list(&plc_name_input);
                                    let plc_ip_input = dialog::input_default(
                                        "Enter PLC IP adress with port",
                                        "10.54.52.201:502",
                                    );
                                    if let Some(plc_ip_input) = plc_ip_input {
                                        psql::postgresql::insert_plc_table(
                                            &plc_name_input,
                                            &plc_name_input,
                                            &plc_ip_input,
                                        );
                                    } else {
                                        psql::postgresql::insert_plc_table(
                                            &plc_name_input,
                                            &plc_name_input,
                                            "10.54.52.201:502",
                                        );
                                    }
                                }
                            }
                            Some(1) => (),
                            Some(_) | None => (),
                        }
                    }
                    DeletePlc => {
                        match self.tree.get_selected_items() {
                            None => {
                                let _warning =
                                    dialog::message_default("Select a PLC in the tree for delete!");
                            }
                            Some(vals) => {
                                let item_depth = vals.as_slice()[0].depth();
                                if item_depth == 1 {
                                    match dialog::choice2_default(
                                        "Do you want to delete this plc?",
                                        "Yes",
                                        "No",
                                        "",
                                    ) {
                                        Some(0) => {
                                            let plc_path =
                                                String::from(vals.as_slice()[0].label().unwrap());
                                            psql::postgresql::delete_plc_table(&plc_path);
                                            psql::postgresql::delete_plc_from_plc_list(&plc_path);
                                            psql::postgresql::delete_plc_from_var_list(&plc_path);
                                            psql::postgresql::delete_plc_from_variables(&plc_path);
                                            self.tree.clear();
                                            add_tree!(self.tree);
                                        }
                                        Some(1) => (),
                                        Some(_) | None => (),
                                    }
                                } else if item_depth == 2 {
                                    let _warning = dialog::message_default(
                                        "Select a plc in the tree for delete!",
                                    );
                                } else {
                                    let _warning = dialog::message_default(
                                        "Something wrong! Select a PLC in the tree to delete!",
                                    );
                                }
                            }
                        };
                    }
                    AddVariable => {
                        match self.tree.get_selected_items() {
                            None => {
                                let _warning = dialog::message_default(
                                    "Select a PLC in the tree to add a new variable!",
                                );
                            }
                            Some(vals) => {
                                let item_depth = vals.as_slice()[0].depth();
                                if item_depth == 1 {
                                    let plc_path =
                                        String::from(vals.as_slice()[0].label().unwrap());
                                    let variable_input = dialog::input_default(
                                        "Enter name variable:",
                                        "temperature",
                                    );
                                    if let Some(variable_input) = variable_input {
                                        let mut path =
                                            String::from(vals.as_slice()[0].label().unwrap());
                                        path.push_str("/");
                                        path.push_str(&variable_input);
                                        self.tree.add(&path);
                                        psql::postgresql::insert_json_variable_to_var_list(
                                            &plc_path,
                                            &variable_input,
                                        );
                                        let addr = dialog::input_default(
                                            "Enter variable address in format 'dec'",
                                            "00015",
                                        );
                                        if let Some(addr) = addr {
                                            match dialog::choice2_default(
                                                "Select modbus function",
                                                "Read/Write",
                                                "Read",
                                                "Write",
                                            ) {
                                                Some(0) => {
                                                    psql::postgresql::insert_json_new_variable(
                                                        &plc_path,
                                                        &variable_input,
                                                        &addr,
                                                        "read/write",
                                                    );
                                                }
                                                Some(1) => {
                                                    psql::postgresql::insert_json_new_variable(
                                                        &plc_path,
                                                        &variable_input,
                                                        &addr,
                                                        "read",
                                                    );
                                                }
                                                Some(2) => {
                                                    psql::postgresql::insert_json_new_variable(
                                                        &plc_path,
                                                        &variable_input,
                                                        &addr,
                                                        "write",
                                                    );
                                                }
                                                Some(_) | None => {
                                                    psql::postgresql::insert_json_new_variable(
                                                        &plc_path,
                                                        &variable_input,
                                                        &addr,
                                                        "read",
                                                    );
                                                }
                                            }
                                        } else {
                                            psql::postgresql::insert_json_new_variable(
                                                &plc_path,
                                                &variable_input,
                                                "00015",
                                                "read",
                                            );
                                        }
                                    }
                                } else if item_depth == 2 {
                                    let _warning = dialog::message_default(
                                        "Select a plc in the tree to add a new variable!",
                                    );
                                } else {
                                    let _warning = dialog::message_default("Something wrong! Select a PLC in the tree to add a new variable!");
                                }
                            }
                        };
                    }
                    DeleteVariable => {
                        match self.tree.get_selected_items() {
                            None => {
                                let _warning = dialog::message_default(
                                    "Select a variable in the tree for delete!",
                                );
                            }
                            Some(vals) => {
                                let item_depth = vals.as_slice()[0].depth();
                                if item_depth == 2 {
                                    match dialog::choice2_default(
                                        "Do you want to delete this variable?",
                                        "Yes",
                                        "No",
                                        "",
                                    ) {
                                        Some(0) => {
                                            let var_name =
                                                String::from(vals.as_slice()[0].label().unwrap());
                                            psql::postgresql::delete_variable_from_var_list(
                                                &var_name,
                                            );
                                            psql::postgresql::delete_variable_from_variables(
                                                &var_name,
                                            );
                                            self.tree.clear();
                                            add_tree!(self.tree);
                                        }
                                        Some(1) => (),
                                        Some(_) | None => (),
                                    }
                                } else if item_depth == 1 {
                                    let _warning = dialog::message_default(
                                        "Select a variable in the tree for delete!",
                                    );
                                } else {
                                    let _warning = dialog::message_default(
                                        "Something wrong! Select a variable in the tree to delete!",
                                    );
                                }
                            }
                        };
                    }
                    GreyBird => {
                        let widget_theme = WidgetTheme::new(ThemeType::Greybird);
                        widget_theme.apply();
                        psql::postgresql::insert_theme_to_themes("greybird");
                    }
                    Metro => {
                        let widget_theme = WidgetTheme::new(ThemeType::Metro);
                        widget_theme.apply();
                        psql::postgresql::insert_theme_to_themes("metro");
                    }
                    Dark => {
                        let widget_theme = WidgetTheme::new(ThemeType::Dark);
                        widget_theme.apply();
                        psql::postgresql::insert_theme_to_themes("dark");
                    }
                    Run => {
                        println!("Run");
                    }
                    Stop => {
                        println!("Stop");
                    }
                }
            }
        }
    }
}

fn main() {
    psql::postgresql::create_themes_table();
    let args: Vec<_> = std::env::args().collect();
    let mut app = MyApp::new(args);
    app.launch();
}
