//! Tests for `log-fastly` with a mock `Endpoint` implementation.
//!
//! These are adapted from the doctests, and I would have preferred to just run them there. However
//! it's awkward to pull it off there because of the different semantics of `Builder::init()` and
//! the `LoggerGuard` type.

#![cfg(feature = "native-test-stubs")]

mod some_dependency {
    pub fn function_that_logs() {
        log::error!("This shouldn't be written");
    }
}

#[test]
fn simple() {
    let logger = log_fastly::reset_logger(
        log_fastly::Logger::builder()
            .max_level(log::LevelFilter::Warn)
            .default_endpoint("my_endpoint")
            .build()
            .unwrap(),
    );
    log::warn!(target: "my_endpoint", "This will be written to my_endpoint...");
    logger.assert_contains("my_endpoint", "This will be written to my_endpoint...");
    log::info!(target: "my_endpoint", "...but this won't");
    logger.assert_absent("my_endpoint", "...but this won't");
}

#[test]
fn multiple_endpoints() {
    let logger = log_fastly::reset_logger(
        log_fastly::Logger::builder()
            .max_level(log::LevelFilter::Warn)
            .default_endpoint("my_endpoint")
            .endpoint("my_other_endpoint")
            .build()
            .unwrap(),
    );
    log::warn!(target: "my_endpoint", "This will be written to my_endpoint...");
    logger.assert_contains("my_endpoint", "This will be written to my_endpoint...");
    log::warn!(target: "my_other_endpoint", "...but this will be written to my_other_endpoint");
    logger.assert_contains(
        "my_other_endpoint",
        "...but this will be written to my_other_endpoint",
    );
}

#[test]
fn endpoint_level() {
    let logger = log_fastly::reset_logger(
        log_fastly::Logger::builder()
            .max_level(log::LevelFilter::Warn)
            .default_endpoint("my_endpoint")
            .endpoint_level("my_other_endpoint", log::LevelFilter::Trace)
            .endpoint_level("error_only", log::LevelFilter::Error)
            .build()
            .unwrap(),
    );
    log::warn!(target: "my_other_endpoint", "This will be written to my_other_endpoint...");
    logger.assert_contains(
        "my_other_endpoint",
        "This will be written to my_other_endpoint...",
    );
    log::trace!(target: "my_other_endpoint", "...but this won't, because max_level wins");
    logger.assert_absent(
        "my_other_endpoint",
        "...but this won't, because max_level wins",
    );
    log::error!(target: "error_only", "This will be written to error_only...");
    logger.assert_contains("error_only", "This will be written to error_only...");
    log::warn!(target: "error_only", "...but this won't, because the endpoint's level is lower");
    logger.assert_absent(
        "error_only",
        "...but this won't, because the endpoint's level is lower",
    );
}

#[test]
fn level_default() {
    let logger = log_fastly::reset_logger(
        log_fastly::Logger::builder()
            .max_level(log::LevelFilter::Info)
            .default_endpoint("my_endpoint")
            .default_level_endpoint("error_only", log::Level::Error)
            .build()
            .unwrap(),
    );
    log::info!("This will be written to my_endpoint...");
    logger.assert_contains("my_endpoint", "This will be written to my_endpoint...");
    log::warn!(".. and this will too.");
    logger.assert_contains("my_endpoint", ".. and this will too.");
    log::error!("But this will be written to error_only");
    logger.assert_contains("error_only", "But this will be written to error_only");
}

#[test]
fn module_name_collision() {
    let logger = log_fastly::reset_logger(
        log_fastly::Logger::builder()
            .max_level(log::LevelFilter::Info)
            .default_endpoint("my_endpoint")
            // modified from the doctests because this module name is `tests`
            .endpoint("tests")
            .build()
            .unwrap(),
    );
    log::info!("This will be written to tests, even though my_endpoint is the default");
    logger.assert_contains(
        "tests",
        "This will be written to tests, even though my_endpoint is the default",
    );
    log::info!(
        target: "my_endpoint",
        "This will be written to my_endpoint, because the target is explicit",
    );
    logger.assert_contains(
        "my_endpoint",
        "This will be written to my_endpoint, because the target is explicit",
    );
}

mod my_app {
    #[test]
    fn module_name_filters() {
        let logger = log_fastly::reset_logger(
            log_fastly::Logger::builder()
                .max_level(log::LevelFilter::Info)
                .default_endpoint("my_endpoint")
                .filter_module("my_app", log::LevelFilter::Warn)
                .build()
                .unwrap(),
        );
        log::warn!("This will be written to my_endpoint");
        logger.assert_contains("my_endpoint", "This will be written to my_endpoint");
        // This won't emit any log messages, because no patterns match `some_dependency`
        super::some_dependency::function_that_logs();
        logger.assert_absent("my_endpoint", "This shouldn't be written");
    }

    mod my_module {
        pub fn do_a_thing() {
            log::warn!("This won't be written, because this module's max level is Error");
        }
    }

    #[test]
    fn module_name_filter_overlaps() {
        let logger = log_fastly::reset_logger(
            log_fastly::Logger::builder()
                .max_level(log::LevelFilter::Info)
                .default_endpoint("my_endpoint")
                .filter_module("my_app$", log::LevelFilter::Warn)
                .filter_module("my_app::my_module", log::LevelFilter::Error)
                .build()
                .unwrap(),
        );
        log::warn!("This will be written to my_endpoint");
        logger.assert_contains("my_endpoint", "This will be written to my_endpoint");
        // This won't emit any log messages, because "my_app$" doesn't match, and "my_app::my_module"
        // is limited to Error
        my_module::do_a_thing();
    }
}
