//! Voter which locks up governance tokens for a user-provided duration in exchange for increased voting power.
#![deny(rustdoc::all)]
#![allow(rustdoc::missing_doc_code_examples)]

pub mod macros;

use anchor_lang::prelude::*;
use anchor_spl::token::{Mint, Token, TokenAccount};
use govern::Governor;
use vipers::*;

pub mod account_structs;
mod events;
mod instructions;
pub mod locker;
mod state;

use account_structs::*;
pub use events::*;
pub use state::*;

declare_id!("VETi1zHeH8qcWfDRyUzUvEZ7DFWs46ETNYMZjMLivHz");

/// Locked voter program.
#[deny(missing_docs)]
#[program]
pub mod locked_voter {
    use super::*;

    /// Creates a new [Locker].
    #[access_control(ctx.accounts.validate())]
    pub fn new_locker(ctx: Context<NewLocker>, bump: u8, params: LockerParams) -> ProgramResult {
        ctx.accounts.new_locker(bump, params)
    }

    /// Creates a new [Escrow] for an account.
    ///
    /// A Vote Escrow, or [Escrow] for short, is an agreement between an account (known as the `authority`) and the DAO to
    /// lock up tokens for a specific period of time, in exchange for voting rights
    /// linearly proportional to the amount of votes given.
    #[access_control(ctx.accounts.validate())]
    pub fn new_escrow(ctx: Context<NewEscrow>, bump: u8) -> ProgramResult {
        ctx.accounts.new_escrow(bump)
    }

    /// Stakes `amount` tokens into the [Escrow].
    #[access_control(ctx.accounts.validate())]
    pub fn lock(ctx: Context<Lock>, amount: u64, duration: i64) -> ProgramResult {
        ctx.accounts.lock(amount, duration)
    }

    /// Exits the DAO; i.e., withdraws all staked tokens in an [Escrow] if the [Escrow] is unlocked.
    #[access_control(ctx.accounts.validate())]
    pub fn exit(ctx: Context<Exit>) -> ProgramResult {
        ctx.accounts.exit()
    }

    /// Activates a proposal.
    #[access_control(ctx.accounts.validate())]
    pub fn activate_proposal(ctx: Context<ActivateProposal>) -> ProgramResult {
        ctx.accounts.activate_proposal()
    }

    /// Casts a vote.
    #[access_control(ctx.accounts.validate())]
    pub fn cast_vote(ctx: Context<CastVote>, side: u8) -> ProgramResult {
        ctx.accounts.cast_vote(side)
    }

    /// Set locker params.
    #[access_control(ctx.accounts.validate())]
    pub fn set_locker_params(ctx: Context<SetLockerParams>, params: LockerParams) -> ProgramResult {
        ctx.accounts.set_locker_params(params)
    }
}
